package immibis.chunkloader;

import net.minecraft.src.*;
import net.minecraftforge.common.*;
import immibis.chunkloader.quota.Quota;
import immibis.core.TileBasicInventory;
import immibis.core.TileCombined;

public class TileChunkLoader extends TileBasicInventory implements ISidedInventory {
	public WorldInfo worldInfo;
	
	public String owner;
	public int radius = 0;
	public boolean isServerOwned = false;
	public boolean isActive = true;
	
	private boolean wasPreviouslyInvalidated = false;
	
	public int currentFuelTicks = 1;	// number of ticks the last fuel item lasts for, never 0
	public int remainingFuelTicks = 0;	// number of ticks left from the last fuel item(s), can be >= currentFuelTicks
	
	public int getScaledFuelTime(int max) {
		if(remainingFuelTicks >= currentFuelTicks)
			return max;
		return remainingFuelTicks * max / currentFuelTicks;
	}

	public TileChunkLoader() {
		super(1, "Chunk loader");
	}
	
	private boolean isLoaderActive;
	public void setActive(boolean active) {
		if(active != isLoaderActive) {
			if(active) {
				limitRadius();
				worldInfo.addLoader(this);
			} else
				worldInfo.removeLoader(this);
			isLoaderActive = active;
		}
	}
	
	public boolean canLoaderBeActive() {
		return !DimensionalAnchors.requireFuel || isServerOwned || remainingFuelTicks >= getNumChunks();
	}
	
	private boolean useFuelItem() {
		ItemStack is = getStackInSlot(0);
		
		if(is == null)
			return false;
		
		int value = Fuels.get(is.itemID, is.getItemDamage());
		if(value <= 0)
			return false;
		
		//System.out.println("Adding "+value+" ticks");
		
		currentFuelTicks = value;
		remainingFuelTicks += value;
		
		is.stackSize--;
		if(is.stackSize <= 0)
			is = null;
		
		setInventorySlotContents(0, is);
		
		return true;
	}
	
	@Override
	public void updateEntity() {
		if(worldObj.isRemote)
			return;
		
		DimensionalAnchors.quota.onTick(this); // currently not used, fuel was made a global thing instead
		
		if(DimensionalAnchors.requireFuel) {
			int usedFuel = getNumChunks();
			
			while(remainingFuelTicks < usedFuel && useFuelItem()) {
			}
			
			if(remainingFuelTicks < usedFuel) {
				remainingFuelTicks = 0;
				setActive(false);
			} else {
				remainingFuelTicks -= usedFuel;
				setActive(true);
				//System.out.println("Used "+usedFuel+" ticks, "+remainingFuelTicks+" remaining");
			}
		}
	}
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setString("owner", owner == null ? "" : owner);
		tag.setInteger("radius", radius);
		tag.setBoolean("serverOwned", isServerOwned);
		tag.setInteger("rfuel", remainingFuelTicks);
		tag.setInteger("mfuel", currentFuelTicks);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		owner = tag.getString("owner");
		if(owner != null && owner.equals(""))
			owner = null;
		radius = tag.getInteger("radius");
		isServerOwned = tag.getBoolean("serverOwned");
		remainingFuelTicks = tag.getInteger("rfuel");
		currentFuelTicks = tag.getInteger("mfuel");
		
		if(currentFuelTicks < 1) currentFuelTicks = 1;
	}
	
	@Override
	public boolean onBlockActivated(EntityPlayer ply) {
		if(!ply.worldObj.isRemote)
			ply.openGui(DimensionalAnchors.instance, DimensionalAnchors.requireFuel ? DimensionalAnchors.GUI_CHUNKLOADER_FUELED : DimensionalAnchors.GUI_CHUNKLOADER, worldObj, xCoord, yCoord, zCoord);
		return true;
	}
	
	@Override
	public void validate() {
		super.validate();
		worldInfo = DimensionalAnchors.getWorld(worldObj);
		
		if(canLoaderBeActive()) {
			setActive(true);
			if(wasPreviouslyInvalidated)
				loadChunks();
		} else
			setActive(false);
		
		wasPreviouslyInvalidated = false;
		if(owner != null)
			Logging.onValidated(getLoaderInfo(), "<unknown>");
	}
	@Override
	public void invalidate() {
		super.invalidate();
		if(isLoaderActive)
			worldInfo.delayRemoveLoader(this);
		if(owner != null)
			Logging.onInvalidated(getLoaderInfo(), "<unknown>");
		wasPreviouslyInvalidated = true;
	}
	
	private void loaderChanged() {
		if(isLoaderActive) {
			setActive(false);
			setActive(true);
		}
	}
	
	public void loaderChanged(String byWho) {
		loaderChanged(byWho, false);
	}
	
	public void loaderChanged(String byWho, boolean justAdded) {
		loaderChanged();
		if(justAdded)
			Logging.onAdded(getLoaderInfo(), byWho);
		else
			Logging.onChanged(getLoaderInfo(), byWho);
	}
	
	void loadChunks() {
		IChunkProvider provider = worldObj.getChunkProvider();
		for(ChunkCoordIntPair ccip : getLoaderInfo().getLoadedChunks())
			if(provider.provideChunk(ccip.chunkXPos, ccip.chunkZPos) instanceof EmptyChunk)
				provider.loadChunk(ccip.chunkXPos, ccip.chunkZPos);
	}
	
	public void limitRadius() {
		if(owner == null) {
			radius = -1;
			return;
		}
		
		int max = DimensionalAnchors.instance.getMaxQuota(owner);
		
		if(max == Quota.UNLIMITED)
			return;
		
		int cur = DimensionalAnchors.instance.getCurQuota(owner);
		
		while(radius >= 0 && cur + getNumChunks() > max) {
			radius--;
			loaderChanged("<enforcing quota limit for "+owner+">");
		}
		
		if(cur + getNumChunks() > max)
			radius--;
	}
	
	public int getNumChunks() {
		if(radius < 0)
			return 0;
		return (2*radius+1)*(2*radius+1);
	}

	public WorldInfo.LoaderInfo getLoaderInfo() {
		WorldInfo.LoaderInfo li = new WorldInfo.LoaderInfo(new WorldInfo.XYZ(this), worldInfo, owner, radius);
		li.isServerOwned = isServerOwned;
		return li;
	}
	
	public int getStartInventorySide(ForgeDirection side) {return 0;}
    public int getSizeInventorySide(ForgeDirection side) {return DimensionalAnchors.allowFuelPiping ? 1 : 0;}
}
