package immibis.chunkloader;

import forge.ISidedInventory;
import immibis.core.TileBasicInventory;
import java.util.Iterator;
import net.minecraft.server.ChunkCoordIntPair;
import net.minecraft.server.EmptyChunk;
import net.minecraft.server.EntityHuman;
import net.minecraft.server.IChunkProvider;
import net.minecraft.server.ItemStack;
import net.minecraft.server.NBTTagCompound;

public class TileChunkLoader extends TileBasicInventory implements ISidedInventory
{
    public WorldInfo worldInfo;
    public String owner;
    public int radius = 0;
    public boolean isServerOwned = false;
    public boolean isActive = true;
    private boolean wasPreviouslyInvalidated = false;
    public int currentFuelTicks = 1;
    public int remainingFuelTicks = 0;
    private boolean isLoaderActive;

    public int getScaledFuelTime(int var1)
    {
        return this.remainingFuelTicks >= this.currentFuelTicks ? var1 : this.remainingFuelTicks * var1 / this.currentFuelTicks;
    }

    public TileChunkLoader()
    {
        super(1, "Chunk loader");
    }

    public void setActive(boolean var1)
    {
        if (var1 != this.isLoaderActive)
        {
            if (var1)
            {
                this.limitRadius();
                this.worldInfo.addLoader(this);
            }
            else
            {
                this.worldInfo.removeLoader(this);
            }

            this.isLoaderActive = var1;
        }
    }

    public boolean canLoaderBeActive()
    {
        return !mod_DimensionalAnchors.requireFuel || this.isServerOwned || this.remainingFuelTicks >= this.getNumChunks();
    }

    private boolean useFuelItem()
    {
        ItemStack var1 = this.getItem(0);

        if (var1 == null)
        {
            return false;
        }
        else
        {
            int var2 = Fuels.get(var1.id, var1.getData());

            if (var2 <= 0)
            {
                return false;
            }
            else
            {
                this.currentFuelTicks = var2;
                this.remainingFuelTicks += var2;
                --var1.count;

                if (var1.count <= 0)
                {
                    var1 = null;
                }

                this.setItem(0, var1);
                return true;
            }
        }
    }

    /**
     * Allows the entity to update its state. Overridden in most subclasses, e.g. the mob spawner uses this to count
     * ticks and creates a new spawn inside its implementation.
     */
    public void q_()
    {
        if (!this.world.isStatic)
        {
            mod_DimensionalAnchors.quota.onTick(this);

            if (mod_DimensionalAnchors.requireFuel)
            {
                int var1 = this.getNumChunks();

                while (this.remainingFuelTicks < var1 && this.useFuelItem())
                {
                    ;
                }

                if (this.remainingFuelTicks < var1)
                {
                    this.remainingFuelTicks = 0;
                    this.setActive(false);
                }
                else
                {
                    this.remainingFuelTicks -= var1;
                    this.setActive(true);
                }
            }
        }
    }

    /**
     * Writes a tile entity to NBT.
     */
    public void b(NBTTagCompound var1)
    {
        super.b(var1);
        var1.setString("owner", this.owner == null ? "" : this.owner);
        var1.setInt("radius", this.radius);
        var1.setBoolean("serverOwned", this.isServerOwned);
        var1.setInt("rfuel", this.remainingFuelTicks);
        var1.setInt("mfuel", this.currentFuelTicks);
    }

    /**
     * Reads a tile entity from NBT.
     */
    public void a(NBTTagCompound var1)
    {
        super.a(var1);
        this.owner = var1.getString("owner");

        if (this.owner != null && this.owner.equals(""))
        {
            this.owner = null;
        }

        this.radius = var1.getInt("radius");
        this.isServerOwned = var1.getBoolean("serverOwned");
        this.remainingFuelTicks = var1.getInt("rfuel");
        this.currentFuelTicks = var1.getInt("mfuel");

        if (this.currentFuelTicks < 1)
        {
            this.currentFuelTicks = 1;
        }
    }

    public boolean onBlockActivated(EntityHuman var1)
    {
        var1.openGui(mod_DimensionalAnchors.instance, mod_DimensionalAnchors.requireFuel ? 1 : 0, this.world, this.x, this.y, this.z);
        return true;
    }

    /**
     * validates a tile entity
     */
    public void m()
    {
        super.m();
        this.worldInfo = mod_DimensionalAnchors.getWorld(this.world);

        if (this.canLoaderBeActive())
        {
            this.setActive(true);

            if (this.wasPreviouslyInvalidated)
            {
                this.loadChunks();
            }
        }
        else
        {
            this.setActive(false);
        }

        this.wasPreviouslyInvalidated = false;

        if (this.owner != null)
        {
            Logging.onValidated(this.getLoaderInfo(), "<unknown>");
        }
    }

    /**
     * invalidates a tile entity
     */
    public void j()
    {
        super.j();

        if (this.isLoaderActive)
        {
            this.worldInfo.delayRemoveLoader(this);
        }

        if (this.owner != null)
        {
            Logging.onInvalidated(this.getLoaderInfo(), "<unknown>");
        }

        this.wasPreviouslyInvalidated = true;
    }

    private void loaderChanged()
    {
        if (this.isLoaderActive)
        {
            this.setActive(false);
            this.setActive(true);
        }
    }

    public void loaderChanged(String var1)
    {
        this.loaderChanged(var1, false);
    }

    public void loaderChanged(String var1, boolean var2)
    {
        this.loaderChanged();

        if (var2)
        {
            Logging.onAdded(this.getLoaderInfo(), var1);
        }
        else
        {
            Logging.onChanged(this.getLoaderInfo(), var1);
        }
    }

    void loadChunks()
    {
        IChunkProvider var1 = this.world.q();
        Iterator var3 = this.getLoaderInfo().getLoadedChunks().iterator();

        while (var3.hasNext())
        {
            ChunkCoordIntPair var2 = (ChunkCoordIntPair)var3.next();

            if (var1.getOrCreateChunk(var2.x, var2.z) instanceof EmptyChunk)
            {
                var1.getChunkAt(var2.x, var2.z);
            }
        }
    }

    public void limitRadius()
    {
        if (this.owner == null)
        {
            this.radius = -1;
        }
        else
        {
            int var1 = mod_DimensionalAnchors.instance.getMaxQuota(this.owner);

            if (var1 != -2)
            {
                int var2 = mod_DimensionalAnchors.instance.getCurQuota(this.owner);

                while (this.radius >= 0 && var2 + this.getNumChunks() > var1)
                {
                    --this.radius;
                    this.loaderChanged("<enforcing quota limit for " + this.owner + ">");
                }

                if (var2 + this.getNumChunks() > var1)
                {
                    --this.radius;
                }
            }
        }
    }

    public int getNumChunks()
    {
        return this.radius < 0 ? 0 : (2 * this.radius + 1) * (2 * this.radius + 1);
    }

    public WorldInfo.LoaderInfo getLoaderInfo()
    {
        WorldInfo.LoaderInfo var1 = new WorldInfo.LoaderInfo(new WorldInfo.XYZ(this), this.worldInfo, this.owner, this.radius);
        var1.isServerOwned = this.isServerOwned;
        return var1;
    }

    public int getStartInventorySide(int var1)
    {
        return 0;
    }

    public int getSizeInventorySide(int var1)
    {
        return mod_DimensionalAnchors.allowFuelPiping ? 1 : 0;
    }
}
