package mods.immibis.infinitubes;

import buildcraft.api.power.IPowerReceptor;
import buildcraft.api.power.PowerHandler;
import buildcraft.api.power.PowerHandler.PowerReceiver;
import ic2.api.energy.event.EnergyTileLoadEvent;
import ic2.api.energy.event.EnergyTileUnloadEvent;
import ic2.api.energy.tile.IEnergySink;
import mods.immibis.core.api.MultiInterfaceClass;
import mods.immibis.core.api.MultiInterfaceClass.Interface;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.world.World;
import net.minecraftforge.common.ForgeDirection;
import net.minecraftforge.common.MinecraftForge;

@MultiInterfaceClass(
	base = "mods.immibis.infinitubes.PowerJunctionTileBase",
	interfaces = {
		@Interface(check="buildcraft.api.power.IPowerReceptor", impl="mods.immibis.infinitubes.PowerJunctionTile$BCImpl"),
		@Interface(check="ic2.api.energy.tile.IEnergySink", impl="mods.immibis.infinitubes.PowerJunctionTile$ICImpl")
	}
)
public class PowerJunctionTile extends PowerJunctionTileBase {
	
	public static class BCImpl extends PowerJunctionTileBase implements IPowerReceptor {
		
		private static final float RATIO = 0.8f; // MJ per item
		private static final int MAX_RATE = 200; // maximum input in MJ/t
		
		private PowerHandler provider;
		
		public BCImpl() {
			provider = new PowerHandler(this, PowerHandler.Type.MACHINE);
			provider.configure(1, MAX_RATE, (int)Math.max(RATIO, 1), (int)(6400*Math.max(RATIO, 1)));
			provider.configurePowerPerdition(0, 0);
		}
		
		@Override
		public void doWork(PowerHandler workProvider) {
			int items = (int)(workProvider.getEnergyStored() / RATIO);
			//System.out.println("a "+items);
			if(items <= 0)
				return;
			
			WorldTubeMap.TubeNet net = getNet();
			if(net == null)
				return;
			
			items = Math.min(items, net.maxStoredPower() - net.storedPower);
			//System.out.println("b "+items);
			if(items <= 0)
				return;
			
			items = (int)(workProvider.useEnergy(RATIO, items * RATIO, true) / RATIO);
			//System.out.println("c "+items);
			if(items <= 0)
				return;
			
			net.storedPower += items;
			WorldTubeMap.getForWorld(worldObj).setDirty(true);
		}
		
		@Override
		public PowerReceiver getPowerReceiver(ForgeDirection side) {
			return provider.getPowerReceiver();
		}
		
		@Override
		public World getWorld() {
			return worldObj;
		}
		
		@Override
		public void updateEntity() {
			super.updateEntity();
			
			if(provider != null)
				provider.update();
		}
		
	}
	
	public static class ICImpl extends PowerJunctionTileBase implements IEnergySink {
		
		private boolean addedToEnet = false;
		
		private int RATIO = 2; // EU per item
		private int MAX_STORED = 2048; // max internal storage (not tubenet storage); also max transferred/tick
		private int storedEU = 0;
		
		@Override
		public void readFromNBT(NBTTagCompound tag) {
			super.readFromNBT(tag);
			storedEU = tag.getInteger("storedEU");
		}
		
		@Override
		public void writeToNBT(NBTTagCompound tag) {
			super.writeToNBT(tag);
			tag.setInteger("storedEU", storedEU);
		}
		
		@Override
		public void invalidate() {
			super.invalidate();
			if(addedToEnet) {
				MinecraftForge.EVENT_BUS.post(new EnergyTileUnloadEvent(this));
				addedToEnet = false;
			}
		}
		
		@Override
		public void onChunkUnload() {
			super.onChunkUnload();
			if(addedToEnet) {
				MinecraftForge.EVENT_BUS.post(new EnergyTileUnloadEvent(this));
				addedToEnet = false;
			}
		}
		
		@Override
		public void updateEntity() {
			if(!addedToEnet && !worldObj.isRemote) {
				MinecraftForge.EVENT_BUS.post(new EnergyTileLoadEvent(this));
				addedToEnet = true;
			}
			super.updateEntity();
			
			if(storedEU > 0) {
				WorldTubeMap.TubeNet net = getNet();
				if(net != null) {
					int maxAdded = Math.min(net.maxStoredPower() - net.storedPower, storedEU / RATIO);
					if(maxAdded > 0) {
						storedEU -= maxAdded * RATIO;
						net.storedPower += maxAdded;
						WorldTubeMap.getForWorld(worldObj).setDirty(true);
					}
				}
			}
		}
		
		@Override
		public boolean acceptsEnergyFrom(TileEntity arg0, ForgeDirection arg1) {
			return true;
		}
		
		@Override
		public double demandedEnergyUnits() {
			return MAX_STORED - storedEU;
		}

		@Override
		public double injectEnergyUnits(ForgeDirection arg0, double arg1) {
			if(storedEU >= MAX_STORED)
				return arg1;
			storedEU += (int)arg1;
			return arg1 - (int)arg1;
		}

		@Override
		public int getMaxSafeInput() {
			return Integer.MAX_VALUE;
		}
		
	}
}
