package mods.immibis.chunkloader.data;

import java.io.File;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;

import mods.immibis.chunkloader.DimensionalAnchors;
import mods.immibis.chunkloader.Logging;
import mods.immibis.chunkloader.Owner;
import mods.immibis.core.api.util.XYZ;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.world.ChunkCoordIntPair;
import net.minecraft.world.World;
import net.minecraft.world.WorldSavedData;
import net.minecraft.world.WorldServer;

public class WorldLoaderList extends WorldSavedData {
	
	private WeakReference<WorldServer> worldRef;
	public Object cliData;
	
	private List<Loader> loaders = new ArrayList<Loader>();
	private LinkedList<DelayedUnloadEntry> delayedUnloadQueue = new LinkedList<DelayedUnloadEntry>();
	private HashMap<ChunkCoordIntPair, Integer> refcounts = new HashMap<ChunkCoordIntPair, Integer>();
	private HashMap<String, Integer> numChunksByOwner = new HashMap<String, Integer>();
	
	// if not -1, then when world time reaches this, all loader info is refreshed
	private long checkTime = -1;
	
	private static class DelayedUnloadEntry {
		public long time; // when time < world time, chunk is unloaded
		public int x;
		public int z;
		
		public DelayedUnloadEntry(long time, int x, int z) {
			this.time = time;
			this.x = x;
			this.z = z;
		}
	}
	
	private void adjustChunksByOwner(String owner, int delta) {
		if(DimensionalAnchors.DEBUG) System.out.println("adjustChunksByOwner "+owner+" "+delta);
		Integer prev = numChunksByOwner.get(owner);
		assert (prev == null ? 0 : prev) + delta >= 0 : "existing count "+prev+", delta "+delta+", new total is negative";
		numChunksByOwner.put(owner, prev == null ? delta : prev+delta);
	}
	
	public WorldServer getWorld() {
		return worldRef == null ? null : worldRef.get();
	}
	
	public WorldLoaderList(String name) {
		super(name);
	}

	public static WorldLoaderList get(WorldServer w) {
		String mapname = "ICL-" + w.provider.getSaveFolder();
		File f = w.getSaveHandler().getMapFileFromName(mapname);
		if(!f.getParentFile().exists())
			if(!f.getParentFile().mkdirs())
				DimensionalAnchors.logger.warning("Failed to create directory: " + f.getParentFile());
		
		WorldLoaderList wi = (WorldLoaderList)w.mapStorage.loadData(WorldLoaderList.class, mapname);
		if(wi == null)
		{
			wi = new WorldLoaderList(mapname);
			wi.worldRef = new WeakReference<WorldServer>(w);
			w.mapStorage.setData(mapname, wi);
		} else {
			wi.worldRef = new WeakReference<WorldServer>(w);
			wi.checkTime = w.getTotalWorldTime() + 40;
			
			// checks removal times are not too far in the past or the future, in case the world time was changed
			// while the server was offline
			long minTime = w.getTotalWorldTime() - 5;
			long maxTime = w.getTotalWorldTime() + 100;
			for(DelayedUnloadEntry e : wi.delayedUnloadQueue)
				if(e.time < minTime)
					e.time = minTime;
				else if(e.time > maxTime)
					e.time = maxTime;
		}
		
		return wi;
	}
	
	public void initialChunkLoad() {
		for(ChunkCoordIntPair chunk : new java.util.ArrayList<ChunkCoordIntPair>(refcounts.keySet()))
			DimensionalAnchors.cli.addChunk(this, chunk);
	}

	@Override
	public void readFromNBT(NBTTagCompound var1) {
		loaders.clear();
		delayedUnloadQueue.clear();
		numChunksByOwner.clear();
		
		{
			NBTTagList list = var1.getTagList("loaders");
			for(int k = 0; k < list.tagCount(); k++) {
				NBTTagCompound c = (NBTTagCompound)list.tagAt(k);
				Loader loader = new Loader(c, this);
				loaders.add(loader);
				if(loader.active) {
					adjustChunksByOwner(loader.owner, loader.coveredChunks.size());
					for(ChunkCoordIntPair ccip : loader.coveredChunks)
						refChunk(ccip);
				}
			}
		}
		
		{
			NBTTagList list = var1.getTagList("duq");
			for(int k = 0; k < list.tagCount(); k++) {
				NBTTagCompound c = (NBTTagCompound)list.tagAt(k);
				long time = c.getLong("time");
				int x = c.getInteger("x");
				int z = c.getInteger("z");
				delayedUnloadQueue.add(new DelayedUnloadEntry(time, x, z));
				refChunk(new ChunkCoordIntPair(x, z));
			}
		}
	}

	@Override
	public void writeToNBT(NBTTagCompound var1) {
		{
			NBTTagList list = new NBTTagList();
			for(Loader l : loaders)
				list.appendTag(l.writeNBT());
			var1.setTag("loaders", list);
		}
		
		{
			NBTTagList list = new NBTTagList();
			for(DelayedUnloadEntry e : delayedUnloadQueue) {
				NBTTagCompound c = new NBTTagCompound();
				c.setLong("time", e.time);
				c.setInteger("x", e.x);
				c.setInteger("z", e.z);
				list.appendTag(c);
			}
			var1.setTag("duq", list);
		}
	}
	
	public void refChunk(ChunkCoordIntPair pos) {
		if(DimensionalAnchors.DEBUG) System.out.println("ref "+pos);
		Integer i = refcounts.get(pos);
		if(i == null) {
			refcounts.put(pos, 1);
			if(getWorld() != null) // world is null while reading from nbt
				DimensionalAnchors.cli.addChunk(this, pos);
		} else
			refcounts.put(pos, i+1);
	}
	
	public void unrefChunk(ChunkCoordIntPair pos) {
		if(DimensionalAnchors.DEBUG) System.out.println("unref "+pos);
		Integer i = refcounts.get(pos);
		if(i == null)
			throw new AssertionError("unref with refcount already 0! chunk pos "+pos);
		else if(i == 1) {
			refcounts.remove(pos);
			DimensionalAnchors.cli.removeChunk(this, pos);
		} else
			refcounts.put(pos, i-1);
	}
	
	// not used
	/*public void removeLoader(Loader loader) {
		Logging.onRemove(loader);
		
		if(loader.active) {
			adjustChunksByOwner(loader.owner, -loader.coveredChunks.size());
			for(ChunkCoordIntPair chunk : loader.coveredChunks)
				unrefChunk(chunk);
		}
		
		loaders.remove(loader);
	}*/
	
	public void delayRemoveLoader(Loader loader) {
		Logging.onDelayRemove(loader);
		
		World world = getWorld();
		if(world == null)
			throw new RuntimeException("world is null");
		
		long time = world.getTotalWorldTime() + 20;
		
		if(loader.active) {
			adjustChunksByOwner(loader.owner, -loader.coveredChunks.size());
			for(ChunkCoordIntPair chunk : loader.coveredChunks)
				delayedUnloadQueue.add(new DelayedUnloadEntry(time, chunk.chunkXPos, chunk.chunkZPos));
		}
		
		loaders.remove(loader);
		
		setDirty(true);
	}
	
	public void addLoader(Loader loader) {
		Logging.onAdd(loader);
		
		loader.onActiveStatePossiblyChanged(false);
		
		World world = getWorld();
		if(world == null)
			throw new RuntimeException("world is null");
		
		loaders.add(loader);
		
		if(loader.active) {
			adjustChunksByOwner(loader.owner, loader.coveredChunks.size());
			for(ChunkCoordIntPair chunk : loader.coveredChunks)
				refChunk(chunk);
		}
		
		setDirty(true);
	}
	
	public void activateLoader(Loader loader) {
		if(loader.active) return;
		
		loader.active = true;
		
		adjustChunksByOwner(loader.owner, loader.coveredChunks.size());
		for(ChunkCoordIntPair chunk : loader.coveredChunks)
			refChunk(chunk);
		
		setDirty(true);
	}
	
	public void deactivateLoader(Loader loader) {
		if(!loader.active) return;
		
		loader.active = false;
		
		adjustChunksByOwner(loader.owner, -loader.coveredChunks.size());
		for(ChunkCoordIntPair chunk : loader.coveredChunks)
			unrefChunk(chunk);
		
		setDirty(true);
	}

	public void tick() {
		World world = getWorld();
		if(world == null)
			return;
		
		long curTime = world.getTotalWorldTime();
		
		while(true) {
			DelayedUnloadEntry e = delayedUnloadQueue.peek();
			if(e == null || e.time > curTime)
				break;
			
			delayedUnloadQueue.removeFirst();
			unrefChunk(new ChunkCoordIntPair(e.x, e.z));
		}
		
		// Rebuild the loaded chunks and loaders list a short time after loading a world
		/*if(checkTime != -1 && checkTime < world.getTotalWorldTime()) {
			LinkedList<Loader> copy = new LinkedList<Loader>(loaders.values());
			loaders.clear();
			loadedChunks.clear();
			loadersByPlayer.clear();
			checkTime = -1;
			for(Loader li : copy)
			{
				TileEntity te = world.getBlockTileEntity(li.pos.x, li.pos.y, li.pos.z);
				if(te instanceof TileChunkLoader)
					addLoader((TileChunkLoader)te);
			}
		}*/
	}
	
	public Collection<? extends ChunkCoordIntPair> getLoadedChunks() {
		return refcounts.keySet();
	}
	
	public boolean isChunkLoaded(ChunkCoordIntPair pos) {
		return refcounts.get(pos) != null;
	}
	
	public int getUsedChunks(String owner) {
		Integer i = numChunksByOwner.get(owner); 
		return i == null ? 0 : i;
	}
	
	public Collection<Loader> getAllLoaders() {
		return loaders;
	}
	
	public String getName() {
		World world = getWorld();
		if(world == null)
			return "<unknown>";
		
		String folder = world.provider.getSaveFolder();
		if(folder == null)
			return "the overworld";
		else
			return "world "+folder;
	}

	public Loader getOrCreateLoaderAt(int xCoord, int yCoord, int zCoord) {
		XYZ xyz = new XYZ(xCoord, yCoord, zCoord);
		for(Loader l : loaders)
			if(l.pos.equals(xyz))
				return l;
		
		Loader l = new Loader(xyz, this, Owner.DATA_LOST_STRING, Collections.<ChunkCoordIntPair>emptyList());
		addLoader(l);
		return l;
	}

	public void onPlayerLogin(String username) {
		onPlayerStateChange(username, true);
	}
	
	public void onPlayerLogout(String username) {
		onPlayerStateChange(username, false);
	}
	
	private void onPlayerStateChange(String username, boolean online) {
		String owner = Owner.getPlayerOwnerString(username);
		for(Loader l : loaders)
			if(l.getOwner().equals(owner))
				l.updateOnlineOK(online);
	}
}
