package mods.immibis.am2;

import ic2.api.Direction;
import ic2.api.energy.event.EnergyTileLoadEvent;
import ic2.api.energy.event.EnergyTileUnloadEvent;
import ic2.api.energy.tile.IEnergySink;
import ic2.api.item.ElectricItem;
import ic2.api.tile.IWrenchable;
import net.minecraft.entity.EntityLiving;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.packet.Packet;
import net.minecraft.network.packet.Packet132TileEntityData;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.Vec3;
import net.minecraftforge.common.MinecraftForge;
import mods.immibis.core.api.traits.IInventoryTrait;
import mods.immibis.core.api.traits.IInventoryTraitUser;
import mods.immibis.core.api.traits.TraitField;
import mods.immibis.core.api.traits.UsesTraits;
import mods.immibis.core.api.util.Dir;

import mods.immibis.core.BasicInventory;
import mods.immibis.core.ImmibisCore;
import mods.immibis.core.TileCombined;

@UsesTraits
public abstract class TileAM2Base extends TileCombined implements IInventoryTraitUser, IEnergySink, IWrenchable /*, IEnergySource*/ {
	
	@TraitField
	public IInventoryTrait inv;

	private static final int MAX_STORAGE = 10000;
	private int storedEnergy = 0;
	
	private static final int MAX_SPEED = 10000; // GUI overflow if >32767
	private int speed; // 0 to MAX_SPEED
	private int spinUpRate = 4;
	private int spinDownRate = 4; 
	
	private static final int MAX_PROGRESS = 120000;
	private int progress; // 0 to MAX_PROGRESS, increments by speed each tick
	
	private static final int MAX_VOLTAGE = 128;
	
	private int facing = 2;
	private boolean visuallyActive;
	
	
	
	public abstract String getGUIName();
	public abstract String getGUIText(int speed);
	public abstract String getMachineName();
	public abstract int getNumOutputSlots();
	public abstract ItemStack getOutputFor(ItemStack input, boolean adjustInput);
	
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setInteger("energy", storedEnergy);
		tag.setShort("facing", (short)facing);
		tag.setInteger("speed", speed);
		inv.writeToNBT(tag);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		try {storedEnergy = tag.getInteger("energy");} catch(Exception e) {}
		try {facing = tag.getShort("facing");} catch(Exception e) {}
		try {speed = tag.getInteger("speed");} catch(Exception e) {}
		inv.readFromNBT(tag);
	}
	
	@Override
	public Packet getDescriptionPacket() {
		return new Packet132TileEntityData(xCoord, yCoord, zCoord, facing | (visuallyActive ? 8 : 0), null);
	}
	
	@Override
	public void onDataPacket(Packet132TileEntityData packet) {
		facing = packet.actionType & 7;
		visuallyActive = (packet.actionType & 8) != 0;
		worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
	}
	
	
	
	private boolean addedToEnet = false;
	@Override
	public void invalidate() {
		if(worldObj.isRemote) return;
		if(addedToEnet) {
			MinecraftForge.EVENT_BUS.post(new EnergyTileUnloadEvent(this));
			addedToEnet = false;
		}
		super.invalidate();
	}
	@Override
	public void onChunkUnload() {
		if(worldObj.isRemote) return;
		if(addedToEnet) {
			MinecraftForge.EVENT_BUS.post(new EnergyTileUnloadEvent(this));
			addedToEnet = false;
		}
		super.onChunkUnload();
	}
	
	@Override
	public void updateEntity() {
		super.updateEntity();
		
		if(worldObj.isRemote) return;
		
		if(!addedToEnet) {
			MinecraftForge.EVENT_BUS.post(new EnergyTileLoadEvent(this));
			addedToEnet = true;
		}
		
		ItemStack battery = inv.getStackInSlot(1);
		if(battery != null) {
			storedEnergy += ElectricItem.discharge(battery, MAX_STORAGE - storedEnergy, 2, false, false);
			if(battery.stackSize == 0)
				inv.setInventorySlotContents(1, null);
		}
		
		ItemStack input = inv.getStackInSlot(0);
		
		boolean spinUp = false;
		boolean isRunning = false;
		
		do {
			if(input == null || storedEnergy < 16)
				break;
			ItemStack output = getOutputFor(input, false);
			if(output == null)
				break;
			if(!hasSpaceForOutput(output.stackSize, output))
				break;
			
			if(!isRunning) {
				isRunning = true;
				progress += speed;
			}
			
			if(progress >= MAX_PROGRESS) {
				progress -= MAX_PROGRESS;
				
				// consume input
				getOutputFor(input, true);
				if(input.stackSize == 0)
					inv.setInventorySlotContents(0, null);
				
				addOutput(ItemStack.copyItemStack(output));
				continue;
			}
			break;
		} while(true);
			
		if(isRunning) {
			spinUp = true;
			storedEnergy -= 16;
		} else {
			spinUp = worldObj.isBlockIndirectlyGettingPowered(xCoord, yCoord, zCoord);
			progress = 0;
		}
		
		if(spinUp) {
			if(storedEnergy >= 1)
				storedEnergy -= 1;
			else
				spinUp = false;
		}
		
		if(spinUp)
			speed = Math.min(MAX_SPEED, speed + spinUpRate);
		else
			speed = Math.max(0, speed - spinDownRate);
		
		if(visuallyActive != (speed > 0)) {
			visuallyActive = (speed > 0);
			resendDescriptionPacket();
		}
	}
	

	
	private boolean hasSpaceForOutput(int num, ItemStack what) {
		int avail = 0;
		for(int k = getNumOutputSlots() + 1; k >= 2; k--) {
			if(inv.getStackInSlot(k) == null)
				avail += what.getMaxStackSize();
			else if(ImmibisCore.areItemsEqual(inv.getStackInSlot(k), what))
				avail += what.getMaxStackSize() - inv.getStackInSlot(k).stackSize;
		}
		return avail >= num;
	}
	
	private void addOutput(ItemStack what) {
		
		BasicInventory.mergeStackIntoRange(what, (IInventory) this, 2, 2 + getNumOutputSlots());
	}



	private int[] allSlots;
	@Override
	public synchronized int[] getAccessibleSlots(int side) {
		if(allSlots == null) {
			allSlots = new int[getInventorySize()];
			for(int k = 0; k < allSlots.length; k++)
				allSlots[k] = k;
		}
		return allSlots;
	}

	@Override
	public boolean canInsert(int slot, int side, ItemStack stack) {
		if(slot == 1 && side == Dir.PY)
			return false; // can't insert batteries from the top
		return true;
	}

	@Override
	public boolean canInsert(int slot, ItemStack stack) {
		if(slot == 0) // input slot
			return getOutputFor(stack, false) != null;
		if(slot == 1) // battery slot
			return ElectricItem.canUse(stack, 1); // can only insert non-empty batteries
		// output slot
		return false;
	}

	@Override
	public boolean canExtract(int slot, int side, ItemStack stack) {
		if(slot == 0) // input slot
			return false;
		if(slot == 1) // battery slot
			return !ElectricItem.canUse(stack, 1); // can only extract empty batteries
		// output slot
		return true;
	}

	@Override
	public int getInventorySize() {
		return 2 + getNumOutputSlots();
	}

	
	
	@Override
	public boolean acceptsEnergyFrom(TileEntity arg0, Direction arg1) {
		return true;
	}

	@Override
	public boolean isAddedToEnergyNet() {
		return addedToEnet;
	}

	@Override
	public int demandsEnergy() {
		return MAX_STORAGE - storedEnergy;
	}

	@Override
	public int getMaxSafeInput() {
		return MAX_VOLTAGE;
	}

	@Override
	public int injectEnergy(Direction arg0, int arg1) {
		if(arg1 > MAX_VOLTAGE) {
			float boomStrength;
			if(arg1 > 8192)
				boomStrength = 16;
			else if(arg1 > 2048)
				boomStrength = 5;
			else if(arg1 > 512)
				boomStrength = 4;
			else if(arg1 > 128)
				boomStrength = 3;
			else
				boomStrength = 2;
			
			/*if(GregTech_API.sMachineWireFire) {
				MinecraftForge.EVENT_BUS.post(new EnergyTileSourceEvent(this, 32));
				MinecraftForge.EVENT_BUS.post(new EnergyTileSourceEvent(this, 128));
				MinecraftForge.EVENT_BUS.post(new EnergyTileSourceEvent(this, 512));
				MinecraftForge.EVENT_BUS.post(new EnergyTileSourceEvent(this, 2048));
			}*/
			
			worldObj.newExplosion(null, xCoord+0.5, yCoord+0.5, zCoord+0.5, boomStrength, false, true);
			return 0;
		}
		if(storedEnergy >= MAX_STORAGE)
			return arg1;
		
		storedEnergy += arg1;
		return 0;
	}
	
	/*@Override
	public boolean emitsEnergyTo(TileEntity arg0, Direction arg1) {
		return false;
	}
	@Override
	public int getMaxEnergyOutput() {
		return 0;
	}*/
	
	
	
	@Override
	public short getFacing() {
		return (short)facing;
	}

	@Override
	public ItemStack getWrenchDrop(EntityPlayer arg0) {
		return new ItemStack(getBlockType().blockID, 1, getBlockMetadata());
	}

	@Override
	public float getWrenchDropRate() {
		return 1.0f;
	}

	@Override
	public void setFacing(short arg0) {
		facing = arg0;
		resendDescriptionPacket();
	}

	@Override
	public boolean wrenchCanRemove(EntityPlayer arg0) {
		return true;
	}

	@Override
	public boolean wrenchCanSetFacing(EntityPlayer arg0, int arg1) {
		return arg1 > 2 && arg1 != facing;
	}
	

	
	@Override
	public void onPlaced(EntityLiving player, int _) {
		Vec3 look = player.getLook(1.0f);
		
        double absx = Math.abs(look.xCoord);
        double absz = Math.abs(look.zCoord);
        
        if(absx > absz) {
        	if(look.xCoord < 0)
        		facing = Dir.PX;
        	else
        		facing = Dir.NX;
        } else {
        	if(look.zCoord < 0)
        		facing = Dir.PZ;
        	else
        		facing = Dir.NZ;
        }
	}
	
	@Override
	public boolean onBlockActivated(EntityPlayer player) {
		if(!worldObj.isRemote)
			player.openGui(AdvancedMachines.INSTANCE, AdvancedMachines.GUI_PROCESSOR, worldObj, xCoord, yCoord, zCoord);
		return true;
	}
	
	public int getScaledProgress(int i) {
		return progress * i / MAX_PROGRESS;
	}
	
	public int getScaledEnergy(int i) {
		return storedEnergy * i / MAX_STORAGE;
	}
	
	public int getSpeed() {
		return speed;
	}
	
	public boolean isVisuallyActive() {
		return visuallyActive;
	}
}
