package immibis.chunkloader;

import immibis.core.CompatibleBaseMod;
import immibis.core.Config;
import immibis.core.CoreProxy;
import immibis.core.IBlockIDCallback;
import immibis.core.ModInfoReader;
import immibis.core.NonSharedProxy;
import immibis.core.net.IPacket;
import immibis.core.net.IPacketMap;
import immibis.core.net.OneTwoFiveNetworking;
/* $if client$ */
import net.minecraft.client.Minecraft;
import org.lwjgl.opengl.GL11;
/* $else$ */
import net.minecraft.server.MinecraftServer;
/* $endif$ */
import net.minecraft.src.*;
import net.minecraft.src.forge.*;
import java.util.*;

public class Main extends CompatibleBaseMod implements IChunkLoadHandler, IGuiHandler, IPacketMap
/* $if client$ */
	, IRenderWorldLastHandler
/* $endif$ */
{
	
	// IDEAS
	// * Let a redstone signal deactivate the chunk loader, removing it from the quota.
	//   This would mean a player would have to reactivate it manually, in most cases,
	//   since the redstone circuit would become unloaded.
	
	// GUI IDs
	public static final int GUI_CHUNKLOADER = 0;
	
	// Packet IDs
	public static final byte S2C_GUI_UPDATE = 0;
	public static final byte C2S_DATA_REQUEST = 1;
	public static final byte S2C_DATA_RESPONSE = 2;
	
	public static final String CHANNEL =  "immibis.chunkldr";
	
	public static final int MAX_RADIUS = 4; // 9x9 area
	
	public Map<World, WorldInfo> worlds = new HashMap<World, WorldInfo>();
	
	public int maxQuota = 18;
	public boolean showOtherPlayersLoaders = false;
	
	public static boolean DEBUG = false;
	
	// Checks whether the game is still using this World object
	public boolean isWorldCurrent(World w) {
		/* $if client$ */
		return ModLoader.getMinecraftInstance().theWorld == w;
		/* $else$
		return DimensionManager.getWorld(w.worldProvider.worldType) == w;
		$endif$ */
	}
	
	public WorldInfo getWorld(World w) {
		WorldInfo wi = worlds.get(w);
		if(wi != null)
			return wi;
		wi = WorldInfo.get(w);
		worlds.put(w, wi);
		return wi;
	}
	
	public static Main instance;
	
	public Main() {
		instance = this;
	}
	
	public BlockChunkLoader block;

	@Override
	public boolean clientSideRequired() {
		return true;
	}

	@Override
	public boolean serverSideRequired() {
		return false;
	}

	@Override
	public String getVersion() {
		return ModInfoReader.getModInfoField("/immibis/chunkloader/mod_ImmiChunkLoaders.info.txt", "version");
	}
	
	/* $if client$ */
	public KeyBinding seeChunksKey;
	/* $endif$ */

	@Override
	public void load() {
		CoreProxy.RegisterBlockID("chunkloader", new IBlockIDCallback() {
			@Override
			public void registerBlock(int id) {
				block = new BlockChunkLoader(id);
				ModLoader.registerBlock(block, ItemChunkLoader.class);
				
				ModLoader.addRecipe(new ItemStack(block, 1, 0),
					" G ",
					"GIG",
					" G ",
					'G', Item.ingotGold,
					'I', Block.blockGold
				);
			}
		});

		ModLoader.registerTileEntity(TileChunkLoader.class, "immibis.chunkloader.TileChunkLoader");
		ModLoader.setInGameHook(this, true, false);
		
		/* $if client$ */
		seeChunksKey = new KeyBinding("Show force-loaded chunks", org.lwjgl.input.Keyboard.KEY_F9);
		ModLoader.registerKey(this, seeChunksKey, false); 
		MinecraftForgeClient.registerRenderLastHandler(this);
		/* $endif$ */
		
		MinecraftForge.registerChunkLoadHandler(this);
		MinecraftForge.setGuiHandler(this, this);
		OneTwoFiveNetworking.initReceive(this, CHANNEL);
		
		/* $if server$ */
		maxQuota = Config.getInt("chunkloader.maxChunksPerPlayer", 3);
		showOtherPlayersLoaders = !Config.getBoolean("chunkloader.hideOtherPlayersLoadersInF9", true);
		/* $endif$ */
	}
	
	/* $if client$ */
	public boolean showingChunks = false;
	
	public Collection<LoadedChunkDisplay> loadedChunkDisplays = null;
	public Map<WorldInfo.XYZ, Boolean> loaderDisplays = null; // pos -> owned
	
	@Override
	public void keyboardEvent(KeyBinding key) {
		if(key == seeChunksKey) {
			showingChunks = !showingChunks;
			if(showingChunks) {
				OneTwoFiveNetworking.send(CHANNEL, new PacketShowChunksRequest(), null);
			} else {
				loadedChunkDisplays = null;
				loaderDisplays = null;
			}
		}
	}
	/* $endif$ */
	
	@Override
	/* $if client$ */
	public boolean onTickInGame(float f, net.minecraft.client.Minecraft mc) {
	/* $else$
	public boolean onTickInGame(net.minecraft.server.MinecraftServer mc) {
	$endif$ */
		Set<World> toRemove = new HashSet<World>();
		for(Map.Entry<World, WorldInfo> e : worlds.entrySet()) {
			if(!isWorldCurrent(e.getKey())) {
				toRemove.add(e.getKey());
			} else {
				e.getValue().tick();
			}
		}
		for(World w : toRemove)
			worlds.remove(w);
		return true;
	}

	@Override
	public Object getGuiElement(int ID, EntityPlayer player, World world, int x, int y, int z) {
		TileEntity tile = world.getBlockTileEntity(x, y, z);
		if(ID == GUI_CHUNKLOADER)
		/* $if client$ */
			return new GuiChunkLoader(new ContainerChunkLoader(player, (TileChunkLoader)tile)); 
		/* $else$
			return new ContainerChunkLoader(player, (TileChunkLoader)tile);
		$endif$ */
		return null;
	}

	@Override
	public boolean canUpdateEntity(Entity entity) {
		return getWorld(entity.worldObj).isChunkForceLoaded(new ChunkCoordIntPair(entity.chunkCoordX, entity.chunkCoordZ));
	}
	
	@Override
	public boolean canUnloadChunk(Chunk chunk) {
		return !getWorld(chunk.worldObj).isChunkForceLoaded(chunk.getChunkCoordIntPair());
	}
	
	@Override
	public void addActiveChunks(World world, Set<ChunkCoordIntPair> chunkList) {
		chunkList.addAll(getWorld(world).getLoadedChunks());
	}

	public int getCurQuota(String player) {
		int r = 0;
		for(Map.Entry<World, WorldInfo> e : worlds.entrySet()) {
			int _this = e.getValue().getCurQuota(player);
			r += _this;
			//System.out.println(e.getKey()+" " + player + " -> " + _this);
		}
		return r;
	}

	public int getMaxQuota(String player) {
		return NonSharedProxy.isOp(player) ? -2 : maxQuota;
		//return maxQuota;
	}

	public boolean canAddQuota(String player, int i) {
		if(i <= 0)
			return true;
		int max = getMaxQuota(player);
		if(max == -2)
			return true;
		return getCurQuota(player) + i <= max;
	}

	/* $if client$ */
	@Override
	public void onRenderWorldLast(RenderGlobal rg, float partialTicks) {
		if(loadedChunkDisplays == null)
			return;
		
		if(loaderDisplays == null) {
			loaderDisplays = new HashMap<WorldInfo.XYZ, Boolean>();
			for(LoadedChunkDisplay lcd : loadedChunkDisplays) {
				loaderDisplays.put(new WorldInfo.XYZ(lcd.loaderX, lcd.loaderY, lcd.loaderZ), lcd.isOwned);
			}
		}
		
		Minecraft mc = ModLoader.getMinecraftInstance();
		EntityLiving rve = mc.renderViewEntity;
		
		GL11.glDisable(GL11.GL_TEXTURE_2D);
		GL11.glDisable(GL11.GL_DEPTH_TEST);
		GL11.glDisable(GL11.GL_CULL_FACE);
		GL11.glEnable(GL11.GL_BLEND);
		GL11.glPushMatrix();
		GL11.glTranslated(
				-(rve.lastTickPosX + (rve.posX - rve.lastTickPosX) * partialTicks),
				-(rve.lastTickPosY + (rve.posY - rve.lastTickPosY) * partialTicks),
				-(rve.lastTickPosZ + (rve.posZ - rve.lastTickPosZ) * partialTicks));
		Tessellator t = Tessellator.instance;
		
		
		// a fixed distance above the player
		double y = (rve.lastTickPosY + (rve.posY - rve.lastTickPosY) * partialTicks) + 20;
		
		t.startDrawingQuads();
		t.setColorRGBA(255, 0, 0, 80);
		for(LoadedChunkDisplay lcd : loadedChunkDisplays) {
			double x = lcd.chunkX * 16;
			double z = lcd.chunkZ * 16;
			t.addVertex(x, y, z);
			t.addVertex(x+16, y, z);
			t.addVertex(x+16, y, z+16);
			t.addVertex(x, y, z+16);
		}
		t.setColorRGBA(0, 255, 255, 80);
		for(Map.Entry<WorldInfo.XYZ, Boolean> e : loaderDisplays.entrySet()) {
			WorldInfo.XYZ xyz = e.getKey();
			double x = xyz.x;
			double y_ = xyz.y;
			double z = xyz.z;
			if(e.getValue() == Boolean.TRUE)
				// owned by you
				t.setColorRGBA(0, 127, 255, 80);
			else
				// owned by someone else
				t.setColorRGBA(255, 127, 0, 80);
			t.addVertex(x+0, y_+0, z+0);
			t.addVertex(x+0, y_+1, z+0);
			t.addVertex(x+1, y_+1, z+0);
			t.addVertex(x+1, y_+0, z+0);
			
			t.addVertex(x+0, y_+0, z+1);
			t.addVertex(x+0, y_+1, z+1);
			t.addVertex(x+1, y_+1, z+1);
			t.addVertex(x+1, y_+0, z+1);
			
			t.addVertex(x+1, y_+0, z+0);
			t.addVertex(x+1, y_+0, z+1);
			t.addVertex(x+0, y_+0, z+1);
			t.addVertex(x+0, y_+0, z+0);
			
			t.addVertex(x+0, y_+1, z+0);
			t.addVertex(x+1, y_+1, z+0);
			t.addVertex(x+1, y_+1, z+1);
			t.addVertex(x+0, y_+1, z+1);
			
			t.addVertex(x+0, y_+0, z+0);
			t.addVertex(x+0, y_+1, z+0);
			t.addVertex(x+0, y_+1, z+1);
			t.addVertex(x+0, y_+0, z+1);
			
			t.addVertex(x+1, y_+0, z+0);
			t.addVertex(x+1, y_+1, z+0);
			t.addVertex(x+1, y_+1, z+1);
			t.addVertex(x+1, y_+0, z+1);
		}
		t.draw();
		
		GL11.glEnable(GL11.GL_DEPTH_TEST);
		
		double bottom = 0;
		
		t.startDrawing(GL11.GL_LINES);
		t.setColorRGBA(0, 255, 0, 80);
		GL11.glLineWidth(2);
		for(LoadedChunkDisplay lcd : loadedChunkDisplays) {
			double x = lcd.chunkX * 16;
			double z = lcd.chunkZ * 16;
			t.addVertex(x, y, z);
			t.addVertex(x, bottom, z);
			t.addVertex(x+16, y, z);
			t.addVertex(x+16, bottom, z);
			t.addVertex(x+16, y, z+16);
			t.addVertex(x+16, bottom, z+16);
			t.addVertex(x, y, z+16);
			t.addVertex(x, bottom, z+16);
		}
		t.draw();
		
		t.startDrawingQuads();
		t.setColorRGBA(0, 0, 255, 40);
		for(LoadedChunkDisplay lcd : loadedChunkDisplays) {
			if(lcd.chunkX != rve.chunkCoordX || lcd.chunkZ != rve.chunkCoordZ)
				continue;
			double x = lcd.chunkX * 16;
			double z = lcd.chunkZ * 16;
			t.addVertex(x, y, z);
			t.addVertex(x, bottom, z);
			t.addVertex(x+16, bottom, z);
			t.addVertex(x+16, y, z);
			t.addVertex(x, y, z+16);
			t.addVertex(x, bottom, z+16);
			t.addVertex(x+16, bottom, z+16);
			t.addVertex(x+16, y, z+16);
			t.addVertex(x, y, z);
			t.addVertex(x, bottom, z);
			t.addVertex(x, bottom, z+16);
			t.addVertex(x, y, z+16);
			t.addVertex(x+16, y, z);
			t.addVertex(x+16, bottom, z);
			t.addVertex(x+16, bottom, z+16);
			t.addVertex(x+16, y, z+16);
			break;
		}
		t.draw();
		
		GL11.glPopMatrix();
		GL11.glDisable(GL11.GL_BLEND);
		GL11.glEnable(GL11.GL_CULL_FACE);
		GL11.glEnable(GL11.GL_TEXTURE_2D);
	}
	/* $endif$ */

	@Override
	public IPacket createPacket(byte id) {
		if(id == S2C_GUI_UPDATE && NonSharedProxy.CLIENT)
			return new PacketGUIUpdate("", 0, 0, 0, false);
		if(id == C2S_DATA_REQUEST && NonSharedProxy.SERVER)
			return new PacketShowChunksRequest();
		if(id == S2C_DATA_RESPONSE && NonSharedProxy.CLIENT)
			return new PacketShowChunksResponse();
		return null;
	}
}
