package immibis.chunkloader;

import immibis.core.Config;

import java.util.HashMap;
import java.util.Map;

import net.minecraft.src.Block;
import net.minecraft.src.Item;
import net.minecraftforge.common.*;

public class Fuels {
	private static Property property;
	static void setProperty(Property prop) {
		property = prop;
		prop.comment = "comma-separated list of id:meta=number-of-ticks or id=number-of-ticks";
		if(prop.value.equals("")) {
			addDefaults();
			save();
		} else {
			
			try {
				for(String fuel : prop.value.split(",")) {
					String[] id_value = fuel.split("=");
					String[] id_meta = id_value[0].split(":");
					if(id_meta.length == 1)
						set(Integer.parseInt(id_meta[0]), Integer.parseInt(id_value[1]));
					else
						set(Integer.parseInt(id_meta[0]), Integer.parseInt(id_meta[1]), Integer.parseInt(id_value[1]));
				}
			} catch(Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Invalid chunkloader fuel option in config");
			}
		}
	}

	private static class Key {
		public int id;
		public int meta;
		
		public static final int ANY_META = 12345678; // must be outside short range
		
		@Override
		public boolean equals(Object o) {
			if(o instanceof Key) {
				Key k = (Key)o;
				return k.id == id && k.meta == meta;
			}
			return false;
		}
		
		@Override
		public String toString() {
			if(meta == ANY_META)
				return String.valueOf(id);
			return id + ":" + meta;
		}
		
		@Override
		public int hashCode() {
			return id + meta * 4097;
		}
		
		public Key(int id, int meta) {
			this.id = id;
			this.meta = meta;
		}
	}
	
	private static Map<Key, Integer> fuels = new HashMap<Key, Integer>();
	
	private static void save() {
		StringBuilder sb = new StringBuilder();
		for(Map.Entry<Key, Integer> e : fuels.entrySet()) {
			if(sb.length() > 0)
				sb.append(',');
			Key k = e.getKey();
			sb.append(k.id);
			if(k.meta != Key.ANY_META) {
				sb.append(':');
				sb.append(k.meta);
			}
			sb.append('=');
			sb.append(e.getValue());
		}
		property.value = sb.toString();
		Config.save();
	}
	
	private static void addDefaults() {
		set(Item.coal, 0, 10*MINUTES);			// coal				10 minutes
		set(Item.coal, 1, 1*MINUTES);			// charcoal			1 minute
		set(Item.redstone, 1*MINUTES);			// redstone			1 minute
		set(Item.enderPearl, 15*MINUTES);		// ender pearl		15 minutes
		set(Item.magmaCream, 15*MINUTES);		// magma cream		15 minutes 
		set(Item.eyeOfEnder, 30*MINUTES);		// ender eye		30 minutes
		set(Block.dirt, 1*SECONDS);				// dirt				1 second
		set(Block.cobblestone, 1*SECONDS);		// cobblestone		1 second
		set(Block.stone, 3*SECONDS);			// stone			3 seconds
		set(Block.netherrack, 4*SECONDS);		// netherrack		4 seconds
		set(Block.netherBrick, 10*SECONDS);		// nether brick		10 seconds
		set(Item.lightStoneDust, 2*MINUTES);	// glowstone dust	2 minutes
	}
	
	public static int SECONDS = 20;
	public static int MINUTES = 1200;
	public static int HOURS = 72000;
	
	public static void set(int id, int meta, int value) {
		if(value <= 0)
			fuels.remove(new Key(id, meta));
		else
			fuels.put(new Key(id, meta), value);
		save();
	}
	
	public static void set(int id, int value) {
		if(value <= 0)
			fuels.remove(new Key(id, Key.ANY_META));
		else
			fuels.put(new Key(id, Key.ANY_META), value);
		save();
	}
	
	public static void set(Item item, int meta, int value) {set(item.shiftedIndex, meta, value);}
	public static void set(Item item, int value) {set(item.shiftedIndex, value);}
	public static void set(Block block, int meta, int value) {set(block.blockID, meta, value);}
	public static void set(Block block, int value) {set(block.blockID, value);}
	
	private static Key getKey = new Key(0, 0);
	public static int get(int id, int meta) {
		getKey.id = id;
		getKey.meta = meta;
		
		Integer v = fuels.get(getKey);
		if(v != null) return v;
		
		getKey.meta = Key.ANY_META;
		v = fuels.get(getKey);
		if(v != null) return v;
		
		return 0;
	}
	
	
	static void addCommands() {
		DimensionalAnchors.commands.put("setfuel", new Command() {
			@Override
			public void invoke(CommandUser cs, String[] args, int nextarg) {
				if(args.length <= nextarg) {cs.send("\u00a7cNot enough arguments."); return;}
				String idstr = args[nextarg++];
				if(args.length <= nextarg) {cs.send("\u00a7cNot enough arguments."); return;}
				String timestr = args[nextarg++];
				
				String unitstr = "";
				if(args.length > nextarg)
					unitstr = args[nextarg++];
			
				int time;
				int id;
				int meta = 0;
				boolean useMeta = false;
				
				String[] idparts = idstr.split(":");
				if(idparts.length != 1 && idparts.length != 2) {
					cs.send("\u00a7cInvalid ID/metadata: "+idstr);
					return;
				}
				
				try {
					id = Integer.parseInt(idparts[0]);
					if(idparts.length > 1) {
						meta = Integer.parseInt(idparts[1]);
						useMeta = true;
					}
					time = Integer.parseInt(timestr);
				} catch(NumberFormatException e) {
					cs.send("\u00a7cInvalid number");
					return;
				}
				
				if(unitstr.equals("sec"))
					time *= 20;
				else if(unitstr.equals("min"))
					time *= 20 * 60;
				else if(unitstr.equals("hr"))
					time *= 20 * 60 * 60;
				else if(!unitstr.equals("")) {
					cs.send("\u00a7cInvalid time unit: "+unitstr+" - must be sec, min, hr or blank");
					return;
				}
				
				if(useMeta)
					set(id, meta, time);
				else
					set(id, time);
				
				save();
				
				cs.send("\u00a7bChunk-loading time for "+(useMeta ? id+":"+meta : String.valueOf(id))+" set to "+time+" ticks");
				cs.send("\u00a7bThis will not affect currently burning items.");
			}
			
			@Override
			public String getUsage() {
				return "/dimanc setfuel <ID>[:<meta>] <time> [sec|min|hr] - use time=0 to remove a fuel"; 
			}
		});
		
		DimensionalAnchors.commands.put("setpipemode", new Command() {
			@Override
			public void invoke(CommandUser cs, String[] args, int nextarg) {
				if(args.length <= nextarg) {cs.send("\u00a7cNot enough arguments."); return;}
				
				String valstr = args[nextarg++];
				boolean val;
				
				if(valstr.equals("allow"))
					val = true;
				else if(valstr.equals("block"))
					val = false;
				else {
					cs.send("\u00a7cInvalid mode: "+valstr+" - must be allow or block");
					return;
				}
				
				DimensionalAnchors.allowFuelPiping = val;
				if(val)
					cs.send("\u00a7bFuel can now be inserted by pipes until the server restarts.");
				else
					cs.send("\u00a7bFuel cannot be inserted by pipes until the server restarts.");
			}
			
			@Override
			public String getUsage() {
				return "/dimanc setpipemode {allow|block} - temporarily sets whether fuel can be inserted by pipes"; 
			}
		});
	}
}
