package mods.immibis.redlogic.gates;


import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import mods.immibis.core.api.util.Dir;
import mods.immibis.microblocks.api.EnumPosition;
import mods.immibis.microblocks.api.IMicroblockCoverSystem;
import mods.immibis.microblocks.api.Part;
import mods.immibis.microblocks.api.PartType;
import mods.immibis.microblocks.api.util.TileCoverableBase;
import mods.immibis.redlogic.IRedstoneUpdatableTile;
import mods.immibis.redlogic.RedLogicMod;
import mods.immibis.redlogic.Utils;
import net.minecraft.client.renderer.RenderBlocks;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagLong;
import net.minecraft.network.INetworkManager;
import net.minecraft.network.packet.Packet;
import net.minecraft.network.packet.Packet132TileEntityData;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.MovingObjectPosition;
import net.minecraftforge.common.ForgeDirection;
import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

public class GateTile extends TileCoverableBase implements IRedstoneUpdatableTile {
	private EnumGates type; // should never be null
	private GateLogic logic; // null on the client
	private byte side; // side of the block the gate is on
	private byte front; // direction the "front" of the gate is facing
	private boolean isNotStateless;
	private int gateSettings;
	private GateLogic.WithPointer pointer;
	private boolean flipped;
	
	float pointerPos; // rendering only
	float pointerSpeed; // rendering only
	
	@Override
	public Packet getDescriptionPacket() {
		if(type == null)
			return null; // should not happen
		
		Packet132TileEntityData p = new Packet132TileEntityData(xCoord, yCoord, zCoord, 0, new NBTTagCompound());
		p.customParam1.setByteArray("c", getCoverSystem().writeDescriptionBytes());
		p.customParam1.setByte("t", (byte)(type.ordinal() | (flipped ? 0x80 : 0)));
		p.customParam1.setByte("s", side);
		p.customParam1.setByte("f", front);
		p.customParam1.setShort("r", (short)prevRenderState);
		if(pointer != null) {
			p.customParam1.setShort("p", (short)pointer.getPointerPosition());
			p.customParam1.setFloat("P", pointer.getPointerSpeed());
		}
		return p;
	}
	
	@Override
	public void onDataPacket(INetworkManager net, Packet132TileEntityData pkt) {
		getCoverSystem().readDescriptionBytes(pkt.customParam1.getByteArray("c"), 0);
		type = EnumGates.VALUES[pkt.customParam1.getByte("t") & 0x7F];
		side = pkt.customParam1.getByte("s");
		front = pkt.customParam1.getByte("f");
		flipped = (pkt.customParam1.getByte("t") & 0x80) != 0;
		
		prevRenderState = pkt.customParam1.getShort("r") & 0xFFFFF;
		
		if(pkt.customParam1.hasKey("p")) {
			pointerPos = pkt.customParam1.getShort("p");
			pointerSpeed = pkt.customParam1.getFloat("P");
		}
		
		worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
	}
	
	public int getSide() {
		return side;
	}
	
	public GateTile(EnumGates type, int side, int front) {
		if(type == null)
			throw new IllegalArgumentException("type cannot be null");
		this.type = type;
		this.side = (byte)side;
		this.front = (byte)front;
		createLogic();
	}
	
	public GateTile() {
	}
	
	private int toBitfield(byte[] a) {
		if(a.length > 8)
			throw new IllegalArgumentException("array too long");
		int rv = 0;
		for(int k = 0; k < a.length; k++) {
			if(a[k] < 0 || a[k] > 15)
				throw new IllegalArgumentException("element out of range (index "+k+", value "+a[k]+")");
			rv = (rv << 4) | a[k];
		}
		return rv;
	}
	
	private long toBitfield(short[] a) {
		if(a.length > 8)
			throw new IllegalArgumentException("array too long");
		long rv = 0;
		for(int k = 0; k < a.length; k++) {
			if(a[k] < 0 || a[k] > 255)
				throw new IllegalArgumentException("element out of range (index "+k+", value "+a[k]+")");
			rv = (rv << 8) | a[k];
		}
		return rv;
	}
	
	private void fromBitfield(int bf, byte[] a) {
		if(a.length > 8)
			throw new IllegalArgumentException("array too long");
		for(int k = a.length - 1; k >= 0; k--) {
			a[k] = (byte)(bf & 15);
			bf >>= 4;
		}
	}
	
	private void fromBitfield(long bf, short[] a) {
		if(a.length > 8)
			throw new IllegalArgumentException("array too long");
		for(int k = a.length - 1; k >= 0; k--) {
			a[k] = (short)(bf & 255);
			bf >>= 8;
		}
	}
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setByte("type", type == null ? -1 : (byte)type.ordinal());
		tag.setByte("side", side);
		tag.setByte("front", front);
		tag.setLong("outputs", toBitfield(outputs));
		tag.setLong("inputs", toBitfield(inputs));
		tag.setLong("absOutputs", toBitfield(absOutputs));
		tag.setLong("prevAbsOutputs", toBitfield(prevAbsOutputs));
		tag.setLong("prevOutputs", toBitfield(prevOutputs));
		tag.setShort("renderState", (short)renderState);
		tag.setShort("prevRenderState", (short)prevRenderState);
		tag.setBoolean("updatePending", updatePending);
		tag.setShort("gateSettings", (short)gateSettings);
		tag.setBoolean("flipped", flipped);
		if(logic != null && isNotStateless) {
			NBTTagCompound tag2 = new NBTTagCompound();
			logic.write(tag2);
			tag.setTag("logic", tag2);
		}
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		try {
			type = EnumGates.VALUES[tag.getByte("type")];
		} catch(Exception e) {
			type = EnumGates.AND; // shouldn't happen
		}
		side = tag.getByte("side");
		front = tag.getByte("front");
		flipped = tag.getBoolean("flipped");
		
		if(tag.getTag("inputs") instanceof NBTTagLong) {
			fromBitfield(tag.getLong("inputs"), inputs);
			fromBitfield(tag.getLong("outputs"), outputs);
			fromBitfield(tag.getLong("absOutputs"), absOutputs);
			fromBitfield(tag.getLong("prevAbsOutputs"), prevAbsOutputs);
			fromBitfield(tag.getLong("prevOutputs"), prevOutputs);
		}
		
		renderState = tag.getShort("renderState") & 0xFFFF;
		prevRenderState = tag.getShort("prevRenderState") & 0xFFFF;
		
		updatePending = tag.getBoolean("updatePending");
		gateSettings = tag.getShort("gateSettings") & 0xFFFF;
		
		createLogic();
		
		if(logic != null && tag.hasKey("logic"))
			logic.read(tag.getCompoundTag("logic"));
	}
	
	private void createLogic() {
		logic = type.createLogic();
		isNotStateless = !(logic instanceof GateLogic.Stateless);
		if(logic instanceof GateLogic.WithPointer)
			pointer = (GateLogic.WithPointer)logic;
		else
			pointer = null;
	}
	
	private boolean isFirstTick = true;
	
	@Override
	public void updateEntity() {
		super.updateEntity();
		
		if(!worldObj.isRemote) {
			if(type == null) {
				System.err.println("RedLogic: invalid gate (no type) at "+xCoord+","+yCoord+","+zCoord+" removed");
				IMicroblockCoverSystem imcs = getCoverSystem();
				if(imcs != null)
					imcs.convertToContainerBlock();
				else
					worldObj.setBlockToAir(xCoord, yCoord, zCoord);
				
			} else if(isNotStateless)
				updateLogic(true, false);
			else if(isFirstTick) {
				updateLogic(false, false);
				isFirstTick = false;
			}
			
		} else {
			pointerPos += pointerSpeed;
		}
	}

	public int getFront() {
		return front;
	}

	public EnumGates getType() {
		return type;
	}
	
	private short[] inputs = new short[4];
	private short[] outputs = new short[4];
	private short[] absOutputs = new short[6];
	private short[] prevAbsOutputs = new short[6];
	private short[] prevOutputs = new short[4];
	private int renderState;
	private int prevRenderState;
	private boolean updatePending;
	
	private void updateRenderState() {
		renderState = logic.getRenderState(inputs, prevOutputs, gateSettings);
		if(prevRenderState != renderState) {
			prevRenderState = renderState;
			worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
		}
	}
	
	private static int[] FLIPMAP_FLIPPED = new int[] {0, 1, 3, 2};
	private static int[] FLIPMAP_UNFLIPPED = new int[] {0, 1, 2, 3};
	
	public void updateLogic(boolean fromTick, boolean forceUpdate) {
		if(type == null)
			return;
		
		int[] flipMap = flipped ? FLIPMAP_FLIPPED : FLIPMAP_UNFLIPPED;
		
		for(int k = 0; k < 4; k++)
			inputs[flipMap[k]] = getInputStrength(Utils.dirMap[side][front][k]);
		
		//if(xCoord == -776)
		//	System.out.println(xCoord+","+yCoord+","+zCoord+" -- "+side+"/"+front+" -- "+Arrays.toString(inputs));
		
		// update render state with new inputs but not new outputs
		updateRenderState();
		
		if(forceUpdate || fromTick == isNotStateless) {
			logic.update(inputs, outputs, gateSettings);

			for(int k = 0; k < 4; k++)
				absOutputs[Utils.dirMap[side][front][k]] = outputs[flipMap[k]];
			absOutputs[side] = absOutputs[side^1] = 0;
			
			if(forceUpdate || !Arrays.equals(outputs, prevOutputs)) {
				
				//if(xCoord == -776)
				//	System.out.println(xCoord+","+yCoord+","+zCoord+" -- "+side+"/"+front+" -- "+Arrays.toString(inputs)+" -> "+Arrays.toString(outputs)+" -> "+Arrays.toString(absOutputs));
				
				if(!updatePending) {
					worldObj.scheduleBlockUpdate(xCoord, yCoord, zCoord, RedLogicMod.gates.blockID, 2);
					updatePending = true;
				}
			}
		}
		
		//System.out.println("in: "+Arrays.toString(inputs)+", out: "+Arrays.toString(outputs)+", out2: "+Arrays.toString(absOutputs));
		//System.out.println("dm: "+Arrays.toString(dirMap[side][front])+", idm: "+Arrays.toString(invDirMap[side][front]));
	}
	
	private short getInputStrength(int dir) {
		switch(dir) {
		case Dir.NX: return Utils.getPowerStrength(worldObj, xCoord-1, yCoord, zCoord, dir^1, side);
		case Dir.PX: return Utils.getPowerStrength(worldObj, xCoord+1, yCoord, zCoord, dir^1, side);
		case Dir.NY: return Utils.getPowerStrength(worldObj, xCoord, yCoord-1, zCoord, dir^1, side);
		case Dir.PY: return Utils.getPowerStrength(worldObj, xCoord, yCoord+1, zCoord, dir^1, side);
		case Dir.NZ: return Utils.getPowerStrength(worldObj, xCoord, yCoord, zCoord-1, dir^1, side);
		case Dir.PZ: return Utils.getPowerStrength(worldObj, xCoord, yCoord, zCoord+1, dir^1, side);
		}
		throw new IllegalArgumentException("Invalid direction "+dir);
	}

	public int getVanillaOutputStrength(int dir) {
		return prevAbsOutputs[dir] / 17;
	}

	public int getRenderState() {
		return prevRenderState;
	}

	public void scheduledTick() {
		updatePending = false;
		//System.out.println(xCoord+","+yCoord+","+zCoord+" Scheduled tick. Outputs: "+Arrays.toString(prevOutputs)+" -> "+Arrays.toString(outputs));
		//System.out.println(xCoord+","+yCoord+","+zCoord+" Scheduled tick. Outputs 2: "+Arrays.toString(prevAbsOutputs)+" -> "+Arrays.toString(absOutputs));
		System.arraycopy(absOutputs, 0, prevAbsOutputs, 0, 6);
		System.arraycopy(outputs, 0, prevOutputs, 0, 4);
		updateRenderState();
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord, RedLogicMod.gates.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord - 1, zCoord, RedLogicMod.gates.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord + 1, zCoord, RedLogicMod.gates.blockID);
        worldObj.notifyBlocksOfNeighborChange(xCoord - 1, yCoord, zCoord, RedLogicMod.gates.blockID);
        worldObj.notifyBlocksOfNeighborChange(xCoord + 1, yCoord, zCoord, RedLogicMod.gates.blockID);
        worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord - 1, RedLogicMod.gates.blockID);
        worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord + 1, RedLogicMod.gates.blockID);
	}
	
	// called when shift-clicked by a screwdriver
	public void configure() {
		if(logic instanceof GateLogic.Flippable) {
			flipped = !flipped;
			worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
		} else
			gateSettings = logic.configure(gateSettings);
		updateLogic(false, true);
	}
	
	// called when non-shift-clicked by a screwdriver
	public void rotate() {
		do
			front = (byte)((front + 1) % 6);
		while((front & 6) == (side & 6));
		
		updateLogic(false, true);
		worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
	}

	public boolean onBlockActivated(EntityPlayer ply) {
		if(ply.getHeldItem() != null && ply.getHeldItem().getItem() == RedLogicMod.screwdriver)
			return false;
		
		if(worldObj.isRemote)
			return type != null && GateLogic.WithRightClickAction.class.isAssignableFrom(type.getLogicClass());
		if(logic instanceof GateLogic.WithRightClickAction) {
			((GateLogic.WithRightClickAction)logic).onRightClick(ply, this);
			return true;
		}
		return false;
	}

	public GateLogic getLogic() {
		return logic;
	}

	public boolean isFlipped() {
		return flipped;
	}

	@Override
	public boolean isPlacementBlockedByTile(PartType<?> type, EnumPosition pos) {
		return !getPartAABBFromPool(0).intersectsWith(Part.getBoundingBoxFromPool(pos, type.getSize()));
	}

	@Override
	public boolean isPositionOccupiedByTile(EnumPosition pos) {
		return pos == EnumPosition.getFacePosition(side);
	}
	
	
	

	@Override
	public EnumPosition getPartPosition(int subHit) {
		if(subHit == 0)
			return EnumPosition.getFacePosition(side);
		return null;
	}

	@Override
	public AxisAlignedBB getPartAABBFromPool(int subHit) {
		if(subHit == 0)
			return Part.getBoundingBoxFromPool(EnumPosition.getFacePosition(side), GateBlock.THICKNESS);
		return null;
	}

	@Override
	protected int getNumTileOwnedParts() {
		return 1;
	}

	@Override
	public float getPlayerRelativePartHardness(EntityPlayer ply, int part) {
		return ply.getCurrentPlayerStrVsBlock(RedLogicMod.gates, false, getBlockMetadata()) / 0.25f / 30f;
	}

	@Override
	public ItemStack pickPart(MovingObjectPosition rayTrace, int part) {
		return new ItemStack(RedLogicMod.gates, 1, getType().ordinal());
	}

	@Override
	public boolean isSolidOnSide(ForgeDirection side) {
		return false;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void render(RenderBlocks render) {
		GateStaticRenderer.instance.renderWorldBlock(render, worldObj, xCoord, yCoord, zCoord, getBlockType(), 0);
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void renderPart(RenderBlocks render, int part) {
		render(render);
	}

	@Override
	public List<ItemStack> removePartByPlayer(EntityPlayer ply, int part) {
		if(cover != null)
			cover.convertToContainerBlock();
		else
			worldObj.setBlock(xCoord, yCoord, zCoord, 0, 0, 3);
		return Collections.singletonList(new ItemStack(RedLogicMod.gates, 1, getType().ordinal()));
	}


	@Override
	public void onRedstoneInputChanged() {
		updateLogic(false, false);
	}
}
