package mods.immibis.redlogic;

import mods.immibis.redlogic.wires.WireTile;

/**
 * Interface implemented by tile entities which connect specially to wires.
 */
public interface IConnectable {
	/**
	 * Called to check whether a wire connects to this block.
	 * @param wire The wire being checked.
	 * @param blockFace The face the wire is on in the block containing it.
	 * @param fromDirection The direction the wire block is in, relative to this block.
	 * @return True to allow the wire connection.
	 */
	public boolean connects(WireTile wire, int blockFace, int fromDirection);

	/**
	 * Called to check whether a wire connects to this block from around a corner (actually an edge)
	 * 
	 * Example:
	 * 
	 * <pre>
	 *   AX
	 *    B
	 * </pre>
	 * In this situation, where A is the wire, X is a solid block and B is this block, X is right and Y is down,
	 * blockFace is -Y and fromDirection is -X.
	 * 
	 * Note this is different from:
	 * <pre>
	 *   A
	 *   XB
	 * </pre>
	 * where blockFace is -X and fromDirection is -Y.
	 * 
	 * @param wire The wire being checked.
	 * @param blockFace The face of *this block* the edge is adjacent to.
	 * @param fromDirection The side the edge is on within that face.
	 * @return True to allow the wire connection.
	 */
	public boolean connectsAroundCorner(WireTile wire, int blockFace, int fromDirection);
}
