package mods.immibis.redlogic.wires;


import java.util.Collections;
import java.util.List;

import mods.immibis.core.api.util.Dir;
import mods.immibis.microblocks.api.EnumPosition;
import mods.immibis.microblocks.api.Part;
import mods.immibis.microblocks.api.PartType;
import mods.immibis.microblocks.api.util.TileCoverableBase;
import mods.immibis.redlogic.IConnectable;
import mods.immibis.redlogic.RedLogicMod;
import mods.immibis.redlogic.Utils;
import net.minecraft.block.Block;
import net.minecraft.client.renderer.RenderBlocks;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.INetworkManager;
import net.minecraft.network.packet.Packet132TileEntityData;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.MovingObjectPosition;
import net.minecraftforge.common.ForgeDirection;
import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

public abstract class WireTile extends TileCoverableBase implements IConnectable {
	
	private EnumWires wireType;
	
	// bitmask where bit (wireSide) is set if there is a wire on that side
	private byte wireMask;
	
	// bitmask where bit ((wireSide * 6) + direction - 2)
	// is set if the wire on that side connects in that direction
	// -1 if not cached (-1 is normally illegal as several bits are always 0)
	// note that bits -2 and -1 are never used, as are bits 32 and 33, so this barely fits
	// into 32 bits (with some unused ones in the middle)
	private int connectMaskCache = -1;
	
	// similar to connectMaskCache but each bit indicates if that connection is around a corner.
	private int connectCornerCache = -1;

	public WireTile(EnumWires type) {
		this.wireType = type;
	}
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		
		tag.setByte("type", (byte)wireType.ordinal());
		tag.setByte("mask", wireMask);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		
		int type = tag.getByte("type");
		if(type < 0 || type >= EnumWires.VALUES.length) {
			wireType = null;
		} else {
			wireType = EnumWires.VALUES[type];
		}
		wireMask = tag.getByte("mask");
	}
	
	@Override
	public abstract boolean canUpdate();
	
	@Override
	public void updateEntity() {
		if(wireType == null) {
			worldObj.setBlock(xCoord, yCoord, zCoord, 0);
			return;
		}
		
		super.updateEntity();
	}

	@Override
	public boolean isPlacementBlockedByTile(PartType type, EnumPosition pos) {
		switch(pos) {
		case FaceNX: return (wireMask & (1 << Dir.NX)) != 0;
		case FacePX: return (wireMask & (1 << Dir.PX)) != 0;
		case FaceNY: return (wireMask & (1 << Dir.NY)) != 0;
		case FacePY: return (wireMask & (1 << Dir.PY)) != 0;
		case FaceNZ: return (wireMask & (1 << Dir.NZ)) != 0;
		case FacePZ: return (wireMask & (1 << Dir.PZ)) != 0;
		default: return false;
		}
	}

	public boolean canPlaceWireOnSide(EnumWires type, int side) {
		int newMask = wireMask | (1 << side);
		return type == wireType && (wireMask & (1 << side)) == 0
			&& newMask != 0x03 && newMask != 0x0C && newMask != 0x30;
	}
	
	void rawAddWire(int side) {
		wireMask |= (byte)(1 << side);
	}

	public boolean addWire(EnumWires type, int side) {
		if(!canPlaceWireOnSide(type, side))
			return false;
		
		wireMask |= (byte)(1 << side);
		worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
		
		computeConnections();
		notifyExtendedNeighbours();
		
		return true;
	}
	
	@Override
	public Packet132TileEntityData getDescriptionPacket() {
		if(wireType == null)
			return null;
		
		if(connectMaskCache == -1 || connectCornerCache == -1)
			computeConnections();
		
		NBTTagCompound tag = new NBTTagCompound();
		
		tag.setByte("t", (byte)wireType.ordinal());
		tag.setByte("m", wireMask);
		tag.setByteArray("c", getCoverSystem().writeDescriptionBytes());
		tag.setInteger("C", connectMaskCache);
		tag.setInteger("C2", connectCornerCache);
		
		return new Packet132TileEntityData(xCoord, yCoord, zCoord, 0, tag);
	}
	
	@Override
	public void onDataPacket(INetworkManager net, Packet132TileEntityData pkt) {
		wireType = EnumWires.VALUES[pkt.customParam1.getByte("t")];
		wireMask = pkt.customParam1.getByte("m");
		getCoverSystem().readDescriptionBytes(pkt.customParam1.getByteArray("c"), 0);
		connectMaskCache = pkt.customParam1.getInteger("C");
		connectCornerCache = pkt.customParam1.getInteger("C2");
		worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
	}

	public final EnumWires getType() {
		return wireType;
	}

	public final byte getSideMask() {
		return wireMask;
	}

	void onNeighbourBlockChange() {
		for(int k = 0; k < 6; k++) {
			if((wireMask & (1 << k)) == 0)
				continue;
			
			int x = xCoord + ForgeDirection.VALID_DIRECTIONS[k].offsetX;
			int y = yCoord + ForgeDirection.VALID_DIRECTIONS[k].offsetY;
			int z = zCoord + ForgeDirection.VALID_DIRECTIONS[k].offsetZ;
			Block b = Block.blocksList[worldObj.getBlockId(x, y, z)];
			if(b == null || !b.isBlockSolidOnSide(worldObj, x, y, z, ForgeDirection.VALID_DIRECTIONS[k ^ 1])) {
				removeAndDropWireOnSide(k);
				if(wireMask == 0)
					return;
			}
		}
		
		computeConnections();
	}
	
	/**
	 * Tile-overridable equivalent of {@link IConnectable#connects(EnumWires, int, int)}. Parameters are the same,
	 * so they are from the perspective of the other block.
	 */
	protected boolean connects(int x, int y, int z, int wireSide, int direction) {
		return false;
	}
	
	/**
	 * Tile-overridable equivalent of {@link IConnectable#connectsAroundCorner(EnumWires, int, int)}. Parameters are the same,
	 * so they are from the perspective of the other block.
	 */
	protected boolean connectsAroundCorner(int x, int y, int z, int wireSide, int direction) {
		return false;
	}
	
	private boolean connectsInDirectionUncached(int wireSide, int direction) {
		if((wireSide & 6) == (direction & 6))
			return false;
		if(!getCoverSystem().isEdgeOpen(wireSide, direction))
			return false;
		if((wireMask & (1 << direction)) != 0)
			return true;
		
		int x = xCoord, y = yCoord, z = zCoord;
		switch(direction) {
		case Dir.NX: x--; break;
		case Dir.PX: x++; break;
		case Dir.NY: y--; break;
		case Dir.PY: y++; break;
		case Dir.NZ: z--; break;
		case Dir.PZ: z++; break;
		default: throw new IllegalArgumentException("invalid direction "+direction);
		}
		if(!worldObj.isAirBlock(x, y, z)) {
			TileEntity te = worldObj.getBlockTileEntity(x, y, z);
			if(te instanceof IConnectable)
				return ((IConnectable)te).connects(wireType, wireSide, direction ^ 1);
			return connects(x, y, z, wireSide, direction ^ 1);
		}
		
		return false;
	}
	
	private boolean connectsAroundCornerUncached(int wireSide, int direction) {
		if((wireSide & 6) == (direction & 6))
			return false;
		if((wireMask & (1 << direction)) != 0)
			return false;
		if(!getCoverSystem().isEdgeOpen(wireSide, direction))
			return false;
		
		int x = xCoord, y = yCoord, z = zCoord;
		switch(direction) {
		case Dir.NX: x--; break;
		case Dir.PX: x++; break;
		case Dir.NY: y--; break;
		case Dir.PY: y++; break;
		case Dir.NZ: z--; break;
		case Dir.PZ: z++; break;
		default: throw new IllegalArgumentException("invalid direction "+direction);
		}
		
		if(!Utils.canConnectThroughEdge(worldObj, x, y, z, wireSide, direction^1)) {
			return false;
		}
		
		switch(wireSide) {
		case Dir.NX: x--; break;
		case Dir.PX: x++; break;
		case Dir.NY: y--; break;
		case Dir.PY: y++; break;
		case Dir.NZ: z--; break;
		case Dir.PZ: z++; break;
		default: throw new IllegalArgumentException("invalid direction "+direction);
		}
		TileEntity te = worldObj.getBlockTileEntity(x, y, z);
		if(te instanceof IConnectable)
			return ((IConnectable)te).connectsAroundCorner(wireType, direction ^ 1, wireSide ^ 1);
		return connectsAroundCorner(x, y, z, direction ^ 1, wireSide ^ 1);
	}
	
	private void computeConnections() {
		int prev = connectMaskCache, prevC = connectCornerCache;
		connectMaskCache = 0;
		connectCornerCache = 0;
		for(int side = 0; side < 6; side++) {
			if((wireMask & (1 << side)) == 0)
				continue;
			for(int dir = 0; dir < 6; dir++) {
				if(connectsInDirectionUncached(side, dir))
					connectMaskCache |= (1 << (side * 6 + dir - 2));
				else if(connectsAroundCornerUncached(side, dir)) {
					connectMaskCache |= (1 << (side * 6 + dir - 2));
					connectCornerCache |= (1 << (side * 6 + dir - 2));
				}
			}
		}
		if(prev != connectMaskCache || prevC != connectCornerCache) {
			worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
			notifyNeighbours();
		}
	}
	
	public boolean connectsInDirection(int wireSide, int direction) {
		if(connectMaskCache == -1)
			computeConnections();
		return (connectMaskCache & (1 << (wireSide * 6 + direction - 2))) != 0;
	}
	
	public boolean connectsInDirectionAroundCorner(int wireSide, int direction) {
		if(connectMaskCache == -1)
			computeConnections();
		return (connectCornerCache & (1 << (wireSide * 6 + direction - 2))) != 0;
	}
	
	public boolean connectsInDirection(int direction) {
		if(connectMaskCache == -1)
			computeConnections();
		// +X    -X    +Z    -Z    +Y    -Y
		// 01000001000001000001000001000001
		//    4   1   0   4   1   0   4   1
		return (connectMaskCache & (0x41041041 << (direction - 2))) != 0;
	}
	
	
	
	
	/** Destroys the wire block when the last wire is removed. */
	public void removeAndDropWireOnSide(int side) {
		if(!removeWireOnSide(side))
			return;
		
		ForgeDirection fd = ForgeDirection.VALID_DIRECTIONS[side];
		
		EntityItem ei = new EntityItem(worldObj,
				xCoord + 0.5 + 0.3*fd.offsetX,
				yCoord + 0.5 + 0.3*fd.offsetY,
				zCoord + 0.5 + 0.3*fd.offsetZ,
				new ItemStack(RedLogicMod.wire, 1, wireType.ordinal()));
		ei.delayBeforeCanPickup = 10;
		worldObj.spawnEntityInWorld(ei);
	}
	
	/** Destroys the wire block when the last wire is removed. */
	public boolean removeWireOnSide(int side) {
		if((wireMask & (1 << side)) == 0)
			return false;
		
		wireMask &= ~(1 << side);
		if(wireMask == 0) {
			if(cover != null)
				cover.convertToContainerBlock();
			else
				worldObj.setBlock(xCoord, yCoord, zCoord, 0, 0, 3);
			notifyExtendedNeighbours();
			
		} else {
			if(wireMask == 0x03)
				removeAndDropWireOnSide(1);
			else if(wireMask == 0x0C)
				removeAndDropWireOnSide(3);
			else if(wireMask == 0x30)
				removeAndDropWireOnSide(5);
			else {
				computeConnections();
				worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
				notifyExtendedNeighbours();
			}
		}
		
		return true;
	}
	
	
	
	
	
	
	
	/** Notifies neighbours one or two blocks away, in the same pattern as most redstone updates. */
	void notifyExtendedNeighbours() {
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord,   RedLogicMod.wire.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord+1, yCoord, zCoord, RedLogicMod.wire.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord-1, yCoord, zCoord, RedLogicMod.wire.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord+1, zCoord, RedLogicMod.wire.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord-1, zCoord, RedLogicMod.wire.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord+1, RedLogicMod.wire.blockID);
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord-1, RedLogicMod.wire.blockID);
	}
	
	private void notifyNeighbourOnClient(int dx, int dy, int dz) {
		if(!worldObj.isRemote)
			throw new IllegalStateException("Not the client.");
		
		int blockID = worldObj.getBlockId(xCoord+dx, yCoord+dy, zCoord+dz);
		if(blockID == RedLogicMod.wire.blockID)
			Block.blocksList[blockID].onNeighborBlockChange(worldObj, xCoord+dx, yCoord+dy, zCoord+dz, RedLogicMod.wire.blockID);
	}
	
	void notifyExtendedNeighbourWiresOnClient() {
		for(int dz = -1; dz <= 1; dz++)
			for(int dy = -1; dy <= 1; dy++)
				for(int dx = -1; dx <= 1; dx++)
					notifyNeighbourOnClient(dx, dy, dz);
		notifyNeighbourOnClient(2, 0, 0);
		notifyNeighbourOnClient(-2, 0, 0);
		notifyNeighbourOnClient(0, 2, 0);
		notifyNeighbourOnClient(0, -2, 0);
		notifyNeighbourOnClient(0, 0, 2);
		notifyNeighbourOnClient(0, 0, -2);
	}
	
	void notifyNeighbours() {
		worldObj.notifyBlocksOfNeighborChange(xCoord, yCoord, zCoord,   RedLogicMod.wire.blockID);
	}
	
	void notifyConnectedWireNeighbours() {
		int notifiedSides = 0;
		
		for(int side = 0; side < 6; side++) {
			for(int dir = 0; dir < 6; dir++) {
				if(connectsInDirection(side, dir)) {
					ForgeDirection fd = ForgeDirection.VALID_DIRECTIONS[dir];
					int x = xCoord + fd.offsetX, y = yCoord + fd.offsetY, z = zCoord + fd.offsetZ;
					
					if(connectsInDirectionAroundCorner(side, dir)) {
						fd = ForgeDirection.VALID_DIRECTIONS[side];
						x += fd.offsetX;
						y += fd.offsetY;
						z += fd.offsetZ;
					
					} else {
						if((notifiedSides & (1 << side)) != 0)
							continue;
						notifiedSides |= 1 << side;
					}
					
					if(worldObj.getBlockId(x, y, z) == RedLogicMod.wire.blockID) {
						RedLogicMod.wire.onNeighborBlockChange(worldObj, x, y, z, RedLogicMod.wire.blockID);
					}
				}
			}
		}
	}
	
	
	
	
	
	
	
	
	
	
	

	@Override
	public boolean connects(EnumWires wireType, int blockFace, int fromDirection) {
		return wireType == this.wireType && (wireMask & (1 << blockFace)) != 0 && getCoverSystem().isEdgeOpen(blockFace, fromDirection);
	}

	@Override
	public boolean connectsAroundCorner(EnumWires wireType, int blockFace, int fromDirection) {
		return connects(wireType, blockFace, fromDirection);
	}

	@Override
	public boolean isPositionOccupiedByTile(EnumPosition pos) {
		switch(pos) {
		case FaceNX: return (wireMask & (1 << Dir.NX)) != 0;
		case FacePX: return (wireMask & (1 << Dir.PX)) != 0;
		case FaceNY: return (wireMask & (1 << Dir.NY)) != 0;
		case FacePY: return (wireMask & (1 << Dir.PY)) != 0;
		case FaceNZ: return (wireMask & (1 << Dir.NZ)) != 0;
		case FacePZ: return (wireMask & (1 << Dir.PZ)) != 0;
		default: return false;
		}
	}
	
	public boolean isWirePresent(int side) {
		return (wireMask & (1 << side)) != 0;
	}
	
	public int getWireColour() {
		return 0xFFFFFF;
	}
	
	
	/*
	 * Returns an array of all connected wires.
	 * If the "array" parameter is not null, it will be filled and then returned instead of creating a new array.
	 * If the "array" parameter is not null, it must reference an array long enough to hold all created wires.
 	 * If the array is longer than the number of connected wires, the extra spaces will be filled with null.
 	 * The maximum possible number of connected wires is no higher than 18.
	 */
	// Not currently used
	/*public WireTile[] getConnectedWires(WireTile[] array) {
		if(connectMaskCache == -1)
			computeConnections();
		
		if(array == null)
			array = new WireTile[Integer.bitCount(connectMaskCache)];
		
		int arraypos = 0;
		
		boolean[] seenDirections = new boolean[6];
		
		for(int side = 0; side < 6; side++) {
			if((wireMask & (1 << side)) == 0)
				continue;
			
			for(int dir = 0; dir < 6; dir++) {
				if(!connectsInDirection(side, dir))
					continue;
				
				ForgeDirection f1 = ForgeDirection.VALID_DIRECTIONS[dir];
				int x = xCoord + f1.offsetX, y = yCoord + f1.offsetY, z = zCoord + f1.offsetZ;
				
				if(connectsInDirectionAroundCorner(side, dir)) {
					ForgeDirection f2 = ForgeDirection.VALID_DIRECTIONS[side];
					x += f2.offsetX; y += f2.offsetY; z += f2.offsetZ;
					
				} else {
					// Only count each non-corner direction once, even if several wire sides connect to it.
					if(seenDirections[dir])
						continue;
					seenDirections[dir] = true;
				}
				
				TileEntity te = worldObj.getBlockTileEntity(x, y, z);
				if(te instanceof WireTile)
					array[arraypos++] = (WireTile)te;
			}
		}
		
		return null;
	}*/

	@Override
	public EnumPosition getPartPosition(int part) {
		return EnumPosition.getFacePosition(part);
	}

	@Override
	public AxisAlignedBB getPartAABBFromPool(int part) {
		if((wireMask & (1 << part)) == 0)
			return null;
		return Part.getBoundingBoxFromPool(EnumPosition.getFacePosition(part), wireType.thickness);
	}

	@Override
	protected int getNumTileOwnedParts() {
		return 6;
	}
	
	@Override
	public float getPlayerRelativePartHardness(EntityPlayer ply, int part) {
		return ply.getCurrentPlayerStrVsBlock(RedLogicMod.wire, false, getType().ordinal()) / 0.25f / 30f;
	}

	@Override
	public ItemStack pickPart(MovingObjectPosition rayTrace, int part) {
		return new ItemStack(RedLogicMod.wire, 1, getType().ordinal());
	}

	@Override
	public boolean isSolidOnSide(ForgeDirection side) {
		return false;
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void render(RenderBlocks render) {
		StaticRenderer.instance.renderWorld(render, getType(), this, getSideMask());
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void renderPart(RenderBlocks render, int part) {
		if((getSideMask() & (1 << part)) == 0)
			return;
		StaticRenderer.instance.renderWorld(render, getType(), this, 1 << part);
	}

	@Override
	public List<ItemStack> removePartByPlayer(EntityPlayer ply, int part) {
		EnumWires type = getType();
		
		if(type == null || (getSideMask() & (1 << part)) == 0)
			return null;
		if(!removeWireOnSide(part))
			return null;
		return Collections.singletonList(new ItemStack(RedLogicMod.wire, 1, type.ordinal()));
	}
}
