package mods.immibis.infinitubes;


import java.util.*;

import mods.immibis.core.api.util.Dir;
import net.minecraft.block.Block;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.world.World;
import net.minecraft.world.WorldSavedData;
import net.minecraftforge.common.ForgeDirection;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

// All access to these objects happens on the server thread.
public class WorldTubeMap extends WorldSavedData {
	public WorldTubeMap(String par1Str) {
		super(par1Str);
	}

	public final static class XYZ {
		public final int x, y, z;
		public XYZ(int x, int y, int z) {
			this.x = x;
			this.y = y;
			this.z = z;
		}
		
		@Override
		public boolean equals(Object o) {
			try {
				XYZ xyz = (XYZ)o;
				return x == xyz.x && y == xyz.y && z == xyz.z;
			} catch(ClassCastException e) {
				return false;
			}
		}
		
		@Override
		public int hashCode() {
			return ((x + 8192) + (z+8192) * 16903) * 256 + y;
		}
		
		@Override
		public String toString() {
			return "["+x+","+y+","+z+"]";
		}

		public XYZ step(int direction) {
			switch(direction) {
			case Dir.NX: return new XYZ(x - 1, y, z);
			case Dir.PX: return new XYZ(x + 1, y, z);
			case Dir.NY: return new XYZ(x, y - 1, z);
			case Dir.PY: return new XYZ(x, y + 1, z);
			case Dir.NZ: return new XYZ(x, y, z - 1);
			case Dir.PZ: return new XYZ(x, y, z + 1);
			default: throw new IllegalArgumentException("Invalid direction "+direction);
			}
		}
	}
	
	public static class TubeNet {
		public final Set<XYZ> tubes = new HashSet<XYZ>();
		public final Multimap<XYZ, ForgeDirection> machineSides = HashMultimap.create();
		
		public int storedPower; // measured in items
		
		public int maxStoredPower() {
			return machineSides.size() * 128;
		}
		
		private final int netID;
		private static int nextID = 0;
		
		public TubeNet() {
			netID = ++nextID;
		}
		
		@Override
		public String toString() {
			return String.valueOf(netID);
		}
	}
	
	
	// FIELDS
	
	private Map<XYZ, TubeNet> tubes = new HashMap<XYZ, TubeNet>();

	private Map<XYZ, Object> machines = new HashMap<XYZ, Object>(); // values are currently not used
	
	
	// PERSISTENCE
	
	public static WorldTubeMap getForWorld(World worldObj) {
		final String DATA_NAME = "infinitubes";
		WorldTubeMap rv = (WorldTubeMap)worldObj.perWorldStorage.loadData(WorldTubeMap.class, DATA_NAME);
		if(rv == null) {
			rv = new WorldTubeMap(DATA_NAME);
			worldObj.perWorldStorage.setData(DATA_NAME, rv);
		}
		return rv;
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		
		tubes.clear();
		machines.clear();
		
		NBTTagList list = tag.getTagList("machines");
		for(int k = 0; k < list.tagCount(); k++) {
			NBTTagCompound machineTag = (NBTTagCompound)list.tagAt(k);
			XYZ xyz = new XYZ(machineTag.getInteger("x"), machineTag.getInteger("y"), machineTag.getInteger("z"));
			machines.put(xyz, null);
		}
		
		list = tag.getTagList("nets");
		for(int k = 0; k < list.tagCount(); k++) {
			NBTTagCompound netTag = (NBTTagCompound)list.tagAt(k);
			
			TubeNet net = new TubeNet();
			
			NBTTagList cablesTag = netTag.getTagList("tubes");
			for(int i = 0; i < cablesTag.tagCount(); i++) {
				NBTTagCompound cableTag = (NBTTagCompound)cablesTag.tagAt(i);
				
				XYZ pos = new XYZ(cableTag.getInteger("x"), cableTag.getInteger("y"), cableTag.getInteger("z"));
				
				net.tubes.add(pos);
				tubes.put(pos, net);
			}
			
			NBTTagList nicsTag = netTag.getTagList("machines");
			for(int i = 0; i < nicsTag.tagCount(); i++) {
				NBTTagCompound nicTag = (NBTTagCompound)nicsTag.tagAt(i);
				
				XYZ pos = new XYZ(nicTag.getInteger("x"), nicTag.getInteger("y"), nicTag.getInteger("z"));
				
				net.machineSides.put(pos, ForgeDirection.VALID_DIRECTIONS[nicTag.getInteger("side")]);
			}
			
			net.storedPower = netTag.getInteger("storedPower");
		}
		
		if(DEBUG) sanityCheck();
	}
	
	private NBTTagCompound xyzToNBT(XYZ xyz) {
		NBTTagCompound tag = new NBTTagCompound();
		tag.setInteger("x", xyz.x);
		tag.setInteger("y", xyz.y);
		tag.setInteger("z", xyz.z);
		return tag;
	}
	
	@Override
	public void writeToNBT(NBTTagCompound root) {
		if(DEBUG) sanityCheck();
		
		NBTTagList machinesTag = new NBTTagList();
		for(Map.Entry<XYZ, Object> e : machines.entrySet()) {
			NBTTagCompound t = xyzToNBT(e.getKey());
			machinesTag.appendTag(t);
		}
		root.setTag("machines", machinesTag);
		
		NBTTagList netsTag = new NBTTagList();
		for(TubeNet net : new HashSet<TubeNet>(tubes.values())) {
			NBTTagCompound netTag = new NBTTagCompound();
			
			NBTTagList cablesTag = new NBTTagList();
			NBTTagList nicsTag = new NBTTagList();
			
			for(XYZ pos : net.tubes)
				cablesTag.appendTag(xyzToNBT(pos));
			
			for(Map.Entry<XYZ, ForgeDirection> pos : net.machineSides.entries()) {
				NBTTagCompound t = xyzToNBT(pos.getKey());
				t.setInteger("side", pos.getValue().ordinal());
				nicsTag.appendTag(t);
			}
			
			netTag.setTag("tubes", cablesTag);
			netTag.setTag("machines", nicsTag);
			
			netTag.setInteger("storedPower", net.storedPower);
			
			netsTag.appendTag(netTag);
		}
		root.setTag("nets", netsTag);
		
	}
	
	
	
	// DEBUGGING
	static final boolean DEBUG = Block.class.getName().equals("net.minecraft.src.Block") && false;
	
	private void sanityCheck() {
		for(Map.Entry<XYZ, TubeNet> e : tubes.entrySet()) {
			if(!e.getValue().tubes.contains(e.getKey()))
				throw new AssertionError("Sanity check failed: Cable's net does not contain cable");
		}
		
		int numCables = 0;
		for(TubeNet e : new HashSet<TubeNet>(tubes.values())) {
			numCables += e.tubes.size();
		}
		
		if(numCables != tubes.size())
			throw new AssertionError("Sanity check failed: Number of cables ("+tubes.size()+") != total net size ("+numCables+")");
	}
	
	
	
	
	
	// MACHINES (all machines are also tubes, currently)

	public void removeMachine(int x, int y, int z) {
		for(ForgeDirection dir : ForgeDirection.VALID_DIRECTIONS) {
			XYZ pos = new XYZ(x + dir.offsetX, y + dir.offsetY, z + dir.offsetZ);
			TubeNet net = tubes.get(pos);
			if(net != null) {
				if(DEBUG)
					System.out.println("removeNIC: Remove NIC "+pos+" from "+net);
				net.machineSides.remove(new XYZ(x, y, z), dir);
			}
		}
		machines.remove(new XYZ(x, y, z));
	}

	public void addMachine(int x, int y, int z) {
		for(ForgeDirection dir : ForgeDirection.VALID_DIRECTIONS) {
			XYZ pos = new XYZ(x + dir.offsetX, y + dir.offsetY, z + dir.offsetZ);
			TubeNet net = tubes.get(pos);
			if(net != null) {
				if(DEBUG)
					System.out.println("addNIC: Add NIC "+pos+" to "+tubes.get(pos));
				net.machineSides.put(new XYZ(x, y, z), dir);
			}
		}
		machines.put(new XYZ(x, y, z), null);
	}
	
	public TubeNet getNet(int x, int y, int z) {
		TubeNet net = tubes.get(new XYZ(x, y, z));
		
		if(DEBUG) {
			System.out.println("Nets:");
			for(Map.Entry<XYZ, TubeNet> e : tubes.entrySet())
				System.out.println(e.getKey()+": "+e.getValue()+": tubes="+e.getValue().tubes+", machineSides="+e.getValue().machineSides);
		}
		
		return net;
	}
	
	
	
	// TUBE NETS
	
	public void addTube(int x, int y, int z) {
		XYZ pos = new XYZ(x, y, z);
		
		// Find an adjacent net to add the new cable to
		TubeNet net = null;
		for(int k = 0; k < 6; k++) {
			XYZ next = pos.step(k);
			TubeNet net2 = tubes.get(next);
			if(net2 != null) {
				if(net == null) {
					if(DEBUG) System.out.println("Using existing net: "+net2);
					net = net2;
				} else
					// If two or more adjacent nets, merge them since they're now connected
					net = mergeNets(net, net2);
			}
		}
		
		// If there was no adjacent net, make a new one
		if(net == null)
			net = new TubeNet();
		
		for(int k = 0; k < 6; k++) {
			XYZ next = pos.step(k);
			if(machines.containsKey(next)) {
				net.machineSides.put(next, ForgeDirection.VALID_DIRECTIONS[k ^ 1]);
			}
		}
		
		if(DEBUG) System.out.println("Adding cable "+pos+" to "+net);
		
		// add cable to net
		tubes.put(pos, net);
		net.tubes.add(pos);
		
		setDirty(true);
	}
	
	public void removeTube(int x, int y, int z) {
		
		XYZ pos = new XYZ(x, y, z);
		
		TubeNet net = tubes.remove(pos);
		if(net == null)
			return;
		net.tubes.remove(pos);
		
		splitNet(net, pos);
		
		setDirty(true);
	}
	
	
	/**
	 * Moves all cables from one net to the other. Returns the merged net.
	 */
	private TubeNet mergeNets(TubeNet net1, TubeNet net2) {
		if(net1 == net2)
			return net1;
		
		if(net1.tubes.size() < net2.tubes.size())
			return mergeNets(net2, net1);
		
		if(DEBUG) System.out.println("Merge net "+net2+" and "+net1);
		
		// Merge net2 into net1
		for(XYZ pos : net2.tubes) {
			tubes.put(pos, net1);
			net1.tubes.add(pos);
			if(DEBUG) System.out.println("  Move cable "+pos+" from "+net2+" to "+net1);
		}
		//if(DEBUG) for(XYZ pos : net2.devices) System.out.println("  Move NIC "+pos+" from "+net2+" to "+net1);
		
		net1.machineSides.putAll(net2.machineSides);
		net2.machineSides.clear();
		net2.tubes.clear();
		
		net1.storedPower += net2.storedPower;
		net2.storedPower = 0;
		
		return net1;
	}

	/**
	 * Splits a net into multiple nets if necessary, after removing the wire at position "pos".
	 */
	private void splitNet(TubeNet net, XYZ pos) {
		if(net == null)
			return;
		
		if(DEBUG) System.out.println("splitNet "+net+" at "+pos);
		
		int oldPower = net.storedPower;
		
		// Find adjacent cable locations
		XYZ neighbours[] = new XYZ[6];
		for(int k = 0; k < 6; k++) {
			XYZ _new = pos.step(k);
			if(net.tubes.contains(_new))
				neighbours[k] = _new;
		}
		
		List<TubeNet> newNets = new ArrayList<TubeNet>(6);
		
		// Build new nets, starting from those locations
		TubeNet newNetsBySide[] = new TubeNet[6];
		for(int k = 0; k < 6; k++) {
			if(neighbours[k] == null)
				continue;
			
			if(newNetsBySide[k] != null)
				continue;
			
			newNetsBySide[k] = new TubeNet();
			addLinkedCablesToNet(newNetsBySide[k], neighbours[k]);
			newNets.add(newNetsBySide[k]);
			
			for(int i = 0; i < 6; i++) {
				if(neighbours[i] == null)
					continue;
				
				if(newNetsBySide[k].tubes.contains(neighbours[i])) {
					if(DEBUG) System.out.println("Side "+i+" is linked to "+k);
					newNetsBySide[i] = newNetsBySide[k];
				}
			}
		}
		
		// split stored power
		
		double totalMax = 0;
		for(TubeNet n : newNets)
			totalMax += n.maxStoredPower();
		if(totalMax > 0) {
			for(TubeNet n : newNets) {
				n.storedPower = (int)(oldPower * n.maxStoredPower() / totalMax);
			}
		}
	}

	/**
	 * Finds all linked cables starting at the given position
	 * and moves them to the given net (including the one at start position).
	 * Ignores cables that are already in the given net.
	 */
	private void addLinkedCablesToNet(TubeNet net, XYZ start) {
		Queue<XYZ> open = new LinkedList<XYZ>();
		
		if(DEBUG)
			System.out.println("addLinkedCablesToNet "+net+" "+start);
		
		if(net.tubes.add(start)) {
			TubeNet net2 = tubes.get(start);
			if(net2 != null)
				net2.tubes.remove(start);

			tubes.put(start, net);
			open.add(start);
			
			if(DEBUG)
				System.out.println("Move cable "+start+" from "+net2+" to "+net);
		}
		
		while(open.size() > 0) {
			XYZ next = open.poll();
			
			// add all neighbours with the same type which are not already in the net
			for(int k = 0; k < 6; k++) {
				XYZ next2 = next.step(k);
				TubeNet net2 = tubes.get(next2);
				if(net2 != null) {
					if(net.tubes.add(next2)) {
						net2.tubes.remove(next2);
						tubes.put(next2, net);
						open.add(next2);
						
						if(DEBUG)
							System.out.println("Move cable "+next2+" from "+net2+" to "+net);
					}
				}
				if(machines.containsKey(next2)) {
					net.machineSides.put(next2, ForgeDirection.VALID_DIRECTIONS[k ^ 1]);
				}
			}
		}
	}
}
