package immibis.core.microblock;

import java.util.List;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.INetworkManager;
import net.minecraft.network.packet.Packet;
import net.minecraft.network.packet.Packet132TileEntityData;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.MovingObjectPosition;
import net.minecraft.util.Vec3;

public abstract class TileCoverableBase extends TileEntity implements ICoverableTile {
	
	protected CoverImpl cover;
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		cover.writeToNBT(tag);
	}
	
	@Override
	public Packet getDescriptionPacket() {
		Packet132TileEntityData p = new Packet132TileEntityData(xCoord, yCoord, zCoord, 0, new NBTTagCompound());
		p.customParam1.setByteArray("C", cover.writeDescriptionBytes());
		return p;
	}
	
	@Override
	public void onDataPacket(INetworkManager net, Packet132TileEntityData pkt) {
		cover.readDescriptionBytes(pkt.customParam1.getByteArray("C"), 0);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		cover.readFromNBT(tag);
	}
	
	public TileCoverableBase(BlockMicroblockContainer wrappedBlock, double hollow_edge_size) {
		cover = new CoverImpl(this, hollow_edge_size);
		cover.wrappedBlock = wrappedBlock;
	}
	
	public boolean isSideOpen(int side) {
		return cover.isSideOpen(side);
	}

	@Override
	public CoverImpl getCoverImpl() {
		return cover;
	}

	public boolean isSideOpen(int l, int i1, int j1) {
		return cover.isSideOpen(xCoord, yCoord, zCoord, l, i1, j1);
	}

	public void convertToMicroblockContainer() {
		if(cover.parts.size() == 0)
			worldObj.setBlockWithNotify(xCoord, yCoord, zCoord, 0);
		else
		{
			worldObj.setBlockWithNotify(xCoord, yCoord, zCoord, CoverSystemProxy.blockMultipart.blockID);
			CoverImpl c2 = ((ICoverableTile)worldObj.getBlockTileEntity(xCoord, yCoord, zCoord)).getCoverImpl();
			for(Part p : cover.parts)
				c2.parts.add(p);
		}
	}
	
	protected abstract int getNumTileOwnedParts();
	
	@Override
	public MovingObjectPosition collisionRayTrace(Vec3 src, Vec3 dst) {
		src = src.addVector(-xCoord, -yCoord, -zCoord);
		dst = dst.addVector(-xCoord, -yCoord, -zCoord);
		
		int numTOP = getNumTileOwnedParts();
		
		MovingObjectPosition best = null;
		double bestDist = 0;
		for(int k = 0; k < numTOP; k++) {
			AxisAlignedBB partBB = getAABBFromPool(k);
			if(partBB == null)
				continue;
			
			MovingObjectPosition _this = partBB.calculateIntercept(src, dst);
			if(_this != null) {
				double dist = _this.hitVec.squareDistanceTo(src);
				if(best == null || dist < bestDist) {
					bestDist = dist;
					best = _this;
					best.subHit = k;
				}
			}
		}
		
		if(best == null)
			return null;
		MovingObjectPosition result = new MovingObjectPosition(xCoord, yCoord, zCoord, best.sideHit, best.hitVec.addVector(xCoord, yCoord, zCoord));
		result.subHit = best.subHit;
		return result;
	}
	
	@Override
	public void getCollidingBoundingBoxes(AxisAlignedBB mask, List<AxisAlignedBB> list) {
		for(int k = 0; k < getNumTileOwnedParts(); k++) {
			AxisAlignedBB partBB = getAABBFromPool(k);
			if(partBB == null)
				continue;
			partBB = partBB.offset(xCoord, yCoord, zCoord);
			if(partBB.intersectsWith(mask))
				list.add(partBB);
		}
	}

}
