package immibis.core.microblock;

import java.util.List;

import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.MovingObjectPosition;
import net.minecraft.util.Vec3;

public interface ICoverableTile {
	public CoverImpl getCoverImpl();

	/**
	 * Returns true if a tile-owned part prevents a part from being placed.
	 * @param type The type of part being placed.
	 * @param pos The position the part is being placed in.
	 * @return False to prevent part placement, otherwise true.
	 */
	public boolean isPlacementBlockedByTile(PartType type, EnumPosition pos);

	/**
	 * Returns true if the given position is occupied by a tile-owned part.
	 * Does not check for collisions with other positions.
	 */
	public boolean isPositionOccupiedByTile(EnumPosition pos);

	/**
	 * Called to ray-trace tile-owned parts.
	 * @param src The start of the ray, in world coordinates.
	 * @param dst The end of the ray, in world coordinates.
	 * @return A ray-trace result, or null if the ray does not intersect any tile-owned parts.
	 */
	public MovingObjectPosition collisionRayTrace(Vec3 src, Vec3 dst);

	/**
	 * Returns the microblock position a tile-owned part mainly occupies, or null if
	 * unknown or the part occupies the whole block.
	 * Used to determine the position of new microblocks placed by the player.
	 * 
	 * @param subHit The tile-owned part ID to check.
	 * @return The position the tile-owned part occupies.
	 */
	public EnumPosition getPosition(int subHit);

	/**
	 * Returns the AABB of a tile-owned part, in tile-local coordinates.
	 * Currently used for drawing the selection box.
	 * May return null, in which case no selection box is drawn.
	 * 
	 * @param subHit The tile-owned part ID to check.
	 * @return The AABB of the tile-owned part, or null.
	 */
	public AxisAlignedBB getAABBFromPool(int subHit);

	/**
	 * Adds the collision boxes of this tile which intersect with "mask" to "list".
	 * All AABBs are in world coordinates.
	 * The boxes can come from the AABB pool.
	 */
	public void getCollidingBoundingBoxes(AxisAlignedBB mask, List<AxisAlignedBB> list);
}
