package immibis.core.microblock;

import immibis.core.api.APILocator;
import immibis.core.api.porting.SidedProxy;
import immibis.core.api.util.Dir;

import java.util.ArrayList;
import java.util.List;
import java.util.WeakHashMap;

import net.minecraft.block.Block;
import net.minecraft.block.BlockContainer;
import net.minecraft.block.material.Material;
import net.minecraft.client.Minecraft;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.Entity;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.EnumMovingObjectType;
import net.minecraft.util.MovingObjectPosition;
import net.minecraft.util.Vec3;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.common.ForgeDirection;
import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

/**
 * When a client player starts breaking a part, or changes the part they are breaking, they send a packet to the server specifying which.
 * When a server player finishes breaking a part, this value is used.
 * 
 * When a player finishes breaking a part, removeBlockByPlayer is called which removes the part but saves it as the last-removed part.
 * If they were using the right tool, harvestBlock is called which drops and clears the last-removed part.
 * 
 * An existing block can be wrapped in covers - in which case the block itself presents one or more "tile-owned" parts
 * with subhit numbers 0 and above. -1 is invalid.
 * (Otherwise, subhit numbers are negative indices into the CoverImpl's parts array, starting at -2)
 * 
 * If the wrapped block is broken, and there are parts remaining, the block is converted into a normal multipart block
 * containing only parts.
 * 
 * Note: A "subhit" refers to the subHit value obtained in a ray trace - it's either -2 minus a part index, -1 (invalid)
 * or a tile-owned part index starting at 0.
 */
public class BlockMicroblockContainer extends BlockContainer implements ICoverableBlock {
	
	
	// Maps players to the subhit value of the cover they are currently breaking.
	// If the player is not currently breaking a cover, their value is undefined.
	private static WeakHashMap<EntityPlayer, Integer> breaking_subhit = new WeakHashMap<EntityPlayer, Integer>();
	
	private final int model;
	
	static AxisAlignedBB selectedBoundingBox = AxisAlignedBB.getBoundingBox(0, 0, 0, 1, 1, 1);
	static int[] texindices = null;
	
	@Override
	public boolean getEnableStats() {return false;}
	
	@SideOnly(Side.CLIENT)
	public AxisAlignedBB getSelectedBoundingBoxFromPoolMultipart(World w, int x, int y, int z) {
		return AxisAlignedBB.getAABBPool().addOrModifyAABBInPool(0, 0, 0, 1, 1, 1).offset(x, y, z);
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public final AxisAlignedBB getSelectedBoundingBoxFromPool(World w, int x, int y, int z) {
		if(selectedBoundingBox == null)
			return getSelectedBoundingBoxFromPoolMultipart(w,x,y,z);
		return selectedBoundingBox.getOffsetBoundingBox(x, y, z);
	}
	
	@SuppressWarnings("rawtypes")
	@Override
	public void addCollidingBlockToList(World world, int x, int y, int z, AxisAlignedBB mask, List list, Entity entity) {
		try {
			ICoverableTile te = ((ICoverableTile)world.getBlockTileEntity(x, y, z));
			te.getCoverImpl().getCollidingBoundingBoxes(world, x, y, z, mask, list);
			te.getCollidingBoundingBoxes(mask, list);
		} catch(ClassCastException e) {
			world.setBlockWithNotify(x, y, z, 0);
		}
	}
	
	@SideOnly(Side.CLIENT)
	public int getBlockTextureFromSideAndMetadataMultipart(int i, int j) {
		return super.getBlockTextureFromSideAndMetadata(i, j);
	}
	
	@SideOnly(Side.CLIENT)
	@Override
	public final int getBlockTextureFromSideAndMetadata(int i, int j) {
		return texindices == null ? getBlockTextureFromSideAndMetadataMultipart(i, j) : texindices[i];
	}
	
	public float getPlayerRelativeBlockHardnessMultipart(EntityPlayer ply, World world, int x, int y, int z, int subhit) {
		return super.getPlayerRelativeBlockHardness(ply, world, x, y, z);
	}
	
	public void harvestBlockMultipart(World world, EntityPlayer ply, int x, int y, int z, int blockMetadata) {
		super.harvestBlock(world, ply, x, y, z, blockMetadata);
	}
	
	private static float hardness;
	//private static boolean forceHarvestState, forcedHarvestState;
	
	public boolean canHarvestBlockMultipart(EntityPlayer ply, int meta) {
		return true;
		//return super.canHarvestBlock(ply, meta);
	}
	
	@Override
	public final boolean canHarvestBlock(EntityPlayer ply, int meta) {
		//if(forceHarvestState)
			//return forcedHarvestState;
		return canHarvestBlockMultipart(ply, meta);
	}
	
	@Override
	public final float getBlockHardness(World w, int x, int y, int z) {
		return hardness;
	}
	
	/**
	 * Returns items to drop (if the player had the correct tool) or null to drop nothing.
	 * @param subhit 
	 */
	public List<ItemStack> removeBlockByPlayerMultipart(World w, EntityPlayer ply, int x, int y, int z, int subhit) {
		List<ItemStack> rv = getBlockDroppedMultipart(w, x, y, z, w.getBlockMetadata(x, y, z), EnchantmentHelper.getFortuneModifier(ply), subhit);
		((TileCoverableBase)w.getBlockTileEntity(x, y, z)).convertToMicroblockContainer();
		return rv;
	}
	
	@Override
	public boolean removeBlockByPlayer(World w, EntityPlayer ply, int x, int y, int z) {
		if(ply.worldObj.isRemote) {
			if(ply.capabilities.isCreativeMode)
				updateBreakingSubhit(x, y, z);
			
			breaking_subhit.remove(ply);
			
			return true; // on the client, do nothing (let the server handle multipart breaking)
			// TODO: client-side prediction of multipart breaking
		}
		
		// remove the part the player was breaking
		int subhit = getBreakingSubhit(ply);
		breaking_subhit.remove(ply);
		
		TileEntity te = w.getBlockTileEntity(x, y, z);
		
		if(!(te instanceof ICoverableTile)) {
			return true;
		}
		
		lastDrop = ((ICoverableTile)te).getCoverImpl().removePartByPlayer(w, ply, x, y, z, subhit);
		
		return true;
	}
	
	private int getBreakingSubhit(EntityPlayer ply) {
		Integer i = breaking_subhit.get(ply);
		if(i == null)
			return -1000;
		return i;
	}
	
	@Override
	@SideOnly(Side.CLIENT)
	public void onBlockClicked(World w, int i, int j, int k, EntityPlayer ply) {
		if(w.isRemote)
			// ensures a PacketMicroblockDigStart will be sent immediately
			breaking_subhit.remove(ply);
	}
	
	@SideOnly(Side.CLIENT)
	private void sendDigStart(int x, int y, int z) {
		APILocator.getNetManager().sendToServer(new PacketMicroblockDigStart(x, y, z, getBreakingSubhit(Minecraft.getMinecraft().thePlayer)));
	}
	
	/**
	 * @param x The X coordinate of the block the player should be breaking.
	 * @param y The Y coordinate of the block the player should be breaking.
	 * @param z The Z coordinate of the block the player should be breaking.
	 * @return True if the player is breaking a valid part of this block. 
	 */
	@SideOnly(Side.CLIENT)
	private boolean updateBreakingSubhit(int x, int y, int z) {
		
		EntityPlayer ply = Minecraft.getMinecraft().thePlayer;
		int oldBSH = getBreakingSubhit(ply);
		
		MovingObjectPosition ray = ply.rayTrace(SidedProxy.instance.getPlayerReach(ply), 0);
		if(ray == null || ray.typeOfHit != EnumMovingObjectType.TILE || ray.blockX != x || ray.blockY != y || ray.blockZ != z) {
			breaking_subhit.remove(ply);
			
		} else {
			breaking_subhit.put(ply, ray.subHit);
			((ICoverableTile)ply.worldObj.getBlockTileEntity(x, y, z)).getCoverImpl().partiallyDamagedPart = ray.subHit;
		}
		
		if(getBreakingSubhit(ply) != oldBSH) {
			sendDigStart(x, y, z);
			resetBreakProgress(ply);
		}
		
		return breaking_subhit.containsKey(ply);
	}
	
	private void resetBreakProgress(EntityPlayer ply) {
		// Need to reset the block damage, but that doesn't seem to be possible
		// Even RP2's covers don't do that
		// TODO: this was not edited since 1.2.5, is it possible now?
		/*PlayerController pc = ModLoader.getMinecraftInstance().playerController;
		pc.resetBlockRemoving();
		pc.updateController();*/
	}
	
	@Override
	public final float getPlayerRelativeBlockHardness(EntityPlayer ply, World world, int x, int y, int z) {
		
		if(world.isRemote)
			updateBreakingSubhit(x, y, z);
		
		int subhit = getBreakingSubhit(ply);
			
		TileEntity te = world.getBlockTileEntity(x, y, z);
		if(te == null || !(te instanceof ICoverableTile))
			return 0.01f;
		else
		{
			CoverImpl cover = ((ICoverableTile)te).getCoverImpl();
			
			if(subhit < 0)
				return getPlayerRelativeBlockHardnessMultipart(ply, world, x, y, z, subhit);
			if(subhit >= cover.parts.size())
				return -1;
			
			Part p = cover.parts.get(subhit);
			{
				hardness = p.type.hardness;
				if(hardness < 0)
					return 0;
				if(!p.type.canHarvestCover(ply))
					return 0.01F / hardness;
				return ply.getCurrentPlayerStrVsBlock(p.type.modelBlock, p.type.modelMeta) / hardness / 30F;
			}
		}
	}
	
	private List<ItemStack> lastDrop = null;
	
	public ArrayList<ItemStack> getBlockDroppedMultipart(World world, int x, int y, int z, int metadata, int fortune, int subhit) {
		return super.getBlockDropped(world, x, y, z, metadata, fortune);
	}

	@Override
	public final ArrayList<ItemStack> getBlockDropped(World world, int x, int y, int z, int metadata, int fortune) {
		if(lastDrop == null)
			return new ArrayList<ItemStack>();
		
		ArrayList<ItemStack> rv = new ArrayList<ItemStack>(lastDrop);
		lastDrop = null;
		return rv;
	}
	
	void setAABB(AxisAlignedBB aabb)
	{
		minX = aabb.minX;
		minY = aabb.minY;
		minZ = aabb.minZ;
		maxX = aabb.maxX;
		maxY = aabb.maxY;
		maxZ = aabb.maxZ;
	}

	protected BlockMicroblockContainer(int id, Material mat) {
		super(id, mat);
		this.model = CoverSystemProxy.coverModel;
	}
	
	
	
	
	@Override
	public final String getTextureFile() {
		return "/terrain.png";
	}
	
	private String wrappedTextureFile;
	public String wrappedGetTextureFile() {
		return wrappedTextureFile;
	}
	
	@Override
	public final Block setTextureFile(String s) {
		wrappedTextureFile = s;
		return this;
	}
	
	
	
	
	/**
	 * Override this for custom collision ray tracing.
	 * Return the ray trace result, or null if nothing intersects the ray.
	 * If the return value is non-null, its {@link MovingObjectPosition#subHit} field should be non-negative.
	 * If the return value is null, the tile entity's collisionRayTrace method is consulted instead.
	 */
	public MovingObjectPosition wrappedCollisionRayTrace(World world, int i, int j, int k, Vec3 vec3d, Vec3 vec3d1) {
		return null;
	}
	
	@Override
	public final MovingObjectPosition collisionRayTrace(World world, int x, int y, int z, Vec3 src, Vec3 dst) {
		try {
			ICoverableTile tile = (ICoverableTile)world.getBlockTileEntity(x, y, z);
			
			MovingObjectPosition ciPos = tile.getCoverImpl().collisionRayTrace(world, x, y, z, src, dst);
			
			MovingObjectPosition tilePos = wrappedCollisionRayTrace(world, x, y, z, src, dst);
			
			if(tilePos != null && tilePos.subHit < 0)
				throw new AssertionError("wrappedCollisionRayTrace must return a non-negative subHit");
			
			if(tilePos == null)
				tilePos = tile.collisionRayTrace(src, dst);
			
			if(tilePos != null && tilePos.subHit < 0)
				throw new AssertionError("IConnectableTile.collisionRayTrace must return a non-negative subHit");
			
			if(tilePos == null) return ciPos;
			if(ciPos == null) return tilePos;
			
			double ciDist = ciPos.hitVec.squareDistanceTo(src);
			double tileDist = tilePos.hitVec.squareDistanceTo(src);
			
			return ciDist < tileDist ? ciPos : tilePos;
			
		} catch(ClassCastException e) {
			world.setBlockWithNotify(x, y, z, 0);
			return super.collisionRayTrace(world, x, y, z, src, dst);
		}
	}
	
	@Override
	public final boolean isOpaqueCube() {
		return false;
	}
	
	@Override
	public final boolean renderAsNormalBlock() {
		return false;
	}
	
	
	
	
	static boolean useWrappedRenderType = false;
	
	/**
	 * Override this for custom rendering of the wrapped block.
	 */
	public int wrappedGetRenderType() {
		return 0;
	}

	@Override
	public final int getRenderType() {
		return useWrappedRenderType ? wrappedGetRenderType() : model;
	}
	
	
	

	@Override
	public TileEntity createNewTileEntity(World world) {
		return new TileMultipart();
	}

	@SuppressWarnings("unchecked")
	public static void setBreakingSubhit(EntityPlayer source, int x, int y, int z, int subhit) {
		breaking_subhit.put(source, subhit);
		
		if(source.capabilities.isCreativeMode) {
			CoverSystemProxy.blockMultipart.removeBlockByPlayer(source.worldObj, source, x, y, z);
		}
		
		for(EntityPlayer pl : (List<EntityPlayer>)source.worldObj.playerEntities)
			APILocator.getNetManager().sendToClient(new PacketUpdateBreakingPart(x, y, z, subhit), pl);
	}
	
	@Override
	public final int getDamageValue(World par1World, int par2, int par3, int par4) {
		// choose some valid damage value; we don't know which part the player was looking at
		return CoverSystemProxy.parts.keySet().iterator().next();
	}

	/**
	 * Override this for custom pick-block behaviour when the wrapped block is picked.
	 */
	public ItemStack getPickBlockMultipart(MovingObjectPosition target, World world, int x, int y, int z, int subHit) {
		return new ItemStack(this, 0, world.getBlockMetadata(x, y, z));
	}
	
	@Override
	public final ItemStack getPickBlock(MovingObjectPosition target, World world, int x, int y, int z) {
		ICoverableTile te = (ICoverableTile)world.getBlockTileEntity(x, y, z);
		if(te == null)
			return null;
		
		CoverImpl ci = te.getCoverImpl();
		if(target.subHit < -1)
			return getPickBlockMultipart(target, world, x, y, z, target.subHit);
		if(target.subHit < 0 || target.subHit >= ci.parts.size())
			return null; // invalid subhit
		return new ItemStack(CoverSystemProxy.blockMultipart.blockID, 1, ci.parts.get(target.subHit).type.id);
	}
	
	@Override
	public boolean shouldSideBeRendered(IBlockAccess par1iBlockAccess, int par2, int par3, int par4, int par5) {
		return true;
	}
	
	@Override
	public boolean isBlockSolidOnSide(World world, int x, int y, int z, ForgeDirection side) {
		CoverImpl ci = ((ICoverableTile)world.getBlockTileEntity(x, y, z)).getCoverImpl();
		switch(side.ordinal()) {
		case Dir.NX: return ci.isPositionOccupied(EnumPosition.FaceNX);
		case Dir.PX: return ci.isPositionOccupied(EnumPosition.FacePX);
		case Dir.NY: return ci.isPositionOccupied(EnumPosition.FaceNY);
		case Dir.PY: return ci.isPositionOccupied(EnumPosition.FacePY);
		case Dir.NZ: return ci.isPositionOccupied(EnumPosition.FaceNZ);
		case Dir.PZ: return ci.isPositionOccupied(EnumPosition.FacePZ);
		}
		return false;
	}

}
