package immibis.chunkloader;

import immibis.chunkloader.porting.ChunkLoadInterface;
import immibis.chunkloader.porting.ChunkLoadInterface132;
import immibis.chunkloader.quota.Quota;
import immibis.core.Config;
import immibis.core.api.APILocator;
import immibis.core.api.FMLModInfo;
import immibis.core.api.IIDCallback;
import immibis.core.api.net.IPacket;
import immibis.core.api.net.IPacketMap;
import immibis.core.api.porting.PortableBaseMod;
import immibis.core.api.porting.SidedProxy;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.logging.Logger;

import net.minecraft.block.Block;
import net.minecraft.command.CommandBase;
import net.minecraft.command.ICommandSender;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.src.ModLoader;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraftforge.common.ConfigCategory;
import net.minecraftforge.common.Configuration;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.Property;
import net.minecraftforge.event.ForgeSubscribe;
import net.minecraftforge.event.world.WorldEvent;
import cpw.mods.fml.common.Mod;
import cpw.mods.fml.common.Mod.Init;
import cpw.mods.fml.common.Mod.PreInit;
import cpw.mods.fml.common.Mod.ServerStarting;
import cpw.mods.fml.common.event.FMLInitializationEvent;
import cpw.mods.fml.common.event.FMLPreInitializationEvent;
import cpw.mods.fml.common.event.FMLServerStartingEvent;
import cpw.mods.fml.common.network.IGuiHandler;
import cpw.mods.fml.common.network.NetworkMod;
import cpw.mods.fml.common.network.NetworkRegistry;

@Mod(version="52.0.6", modid="DimensionalAnchors", name="Dimensional Anchors", dependencies="required-after:ImmibisCore")
@NetworkMod(clientSideRequired = true, serverSideRequired = false)
@FMLModInfo(
	url="http://www.minecraftforum.net/topic/1001131-110-immibiss-mods-smp/",
	description="Yet another chunkloader mod.",
	authors="immibis"
	)
public class DimensionalAnchors extends PortableBaseMod implements IPacketMap {

	// IDEAS
	// * Let a redstone signal deactivate the chunk loader, removing it from the quota.
	//   This would mean a player would have to reactivate it manually, in most cases,
	//   since the redstone circuit would become unloaded.
	// * Let a *wireless* redstone signal do that.

	public static Logger logger = Logger.getLogger("DimensionalAnchors");

	// GUI IDs
	public static final int GUI_CHUNKLOADER = 0;
	public static final int GUI_CHUNKLOADER_FUELED = 1;

	// Packet IDs
	public static final byte S2C_GUI_UPDATE = 0;
	public static final byte C2S_DATA_REQUEST = 1;
	public static final byte S2C_DATA_RESPONSE = 2;

	public static final String CHANNEL =  "immibis.chunkldr";
	public static int MAX_RADIUS = 4; // 9x9 area; now configurable
	public static boolean DEBUG = Block.class.getName().equals("net.minecraft.src.Block") && false;

	public static Map<World, WorldInfo> worlds = new WeakHashMap<World, WorldInfo>();
	public static boolean showOtherPlayersLoaders = false;
	public static ChunkLoadInterface cli;
	public static Quota quota;

	public static boolean requireFuel;
	public static boolean allowFuelPiping;

	public static Map<String, Property> playerQuotaOverride;

	public static @cpw.mods.fml.common.SidedProxy(clientSide="immibis.chunkloader.ClientProxy",serverSide="immibis.chunkloader.BaseProxy") BaseProxy proxy;

	// Checks whether the game is still using this World object
	public boolean isWorldCurrent(World w) {
		return SidedProxy.instance.isWorldCurrent(w);
	}

	public static WorldInfo getWorld(World w) {
		WorldInfo wi = worlds.get(w);
		if(wi != null)
			return wi;
		
		if(!(w instanceof WorldServer))
			throw new IllegalArgumentException("no chunkloading on client worlds");
		
		wi = WorldInfo.get(w);
		worlds.put(w, wi);
		Logging.onLoadWorld(wi);
		cli.onLoadWorld(wi);
		return wi;
	}

	public static DimensionalAnchors instance;

	public DimensionalAnchors() {
		instance = this;
	}

	public BlockChunkLoader block;

	public static HashMap<String, Command> commands = new HashMap<String, Command>();

	@ServerStarting
	public void onServerStart(FMLServerStartingEvent event) {
		event.registerServerCommand(new CommandBase() {
			@Override
			public void processCommand(final ICommandSender var1, String[] var2) {
				CommandUser user = new CommandUser() {
					@Override
					public void send(String s) {
						var1.sendChatToPlayer(s);
					}
				};

				if(var2.length == 0) {
					var1.sendChatToPlayer("\u00a7cFor command list, use /dimanc help");

				} else if(var2[0].equals("help")) {
					if(var2.length < 2) {
						var1.sendChatToPlayer("\u00a7cTry:");
						for(String s : commands.keySet())
							var1.sendChatToPlayer("\u00a7c/dimanc help "+s);

					} else {
						if(commands.containsKey(var2[1]))
							commands.get(var2[1]).sendUsage(user, var2);
						else
							var1.sendChatToPlayer("\u00a7cInvalid command. For command list, use /dimanc help");
					}

				} else if(!commands.containsKey(var2[0])) {
					var1.sendChatToPlayer("\u00a7cInvalid command. For command list, use /dimanc help");
				} else {
					commands.get(var2[0]).invoke(user, var2, 1);
				}
			}

			@Override
			public String getCommandName() {
				return "dimanc";
			}
		});
	}

	@ForgeSubscribe
	public void onWorldUnload(WorldEvent.Unload evt) {
		WorldInfo wi = worlds.remove(evt.world);
		if(wi != null) {
			cli.onUnloadWorld(wi);
			Logging.onUnloadWorld(wi);
		}
	}

	@PreInit
	public void preinit(FMLPreInitializationEvent evt) {
		APILocator.getIDAllocator().requestBlock(this, "chunkloader", new IIDCallback() {

			@Override
			public void register(int id) {
				block = new BlockChunkLoader(id);
				ModLoader.registerBlock(block, ItemChunkLoader.class);
			}
		});

		APILocator.getIDAllocator().addRecipes(new Runnable() {
			@Override
			public void run() {
				if(Config.getBoolean("chunkloader.enableCrafting", true)) {
					ModLoader.addRecipe(new ItemStack(block, 1, 0),
							" G ",
							"GIG",
							" G ",
							'G', Item.ingotGold,
							'I', Block.blockSteel
							);
				}
			}
		});
	}

	@Init
	public void load(FMLInitializationEvent evt) {
		cli = new ChunkLoadInterface132();

		if(cli == null)
			throw new RuntimeException("FIX ME - No ChunkLoadInterface!");

		MinecraftForge.EVENT_BUS.register(this);

		ModLoader.registerTileEntity(TileChunkLoader.class, "immibis.chunkloader.TileChunkLoader");
		enableClockTicks(true);

		proxy.load();

		NetworkRegistry.instance().registerGuiHandler(this, new IGuiHandler() {

			@Override
			public Object getClientGuiElement(int ID, EntityPlayer player, World world, int x, int y, int z) {
				TileEntity tile = world.getBlockTileEntity(x, y, z);
				if(ID == GUI_CHUNKLOADER)
					return new GuiAnchor(new ContainerChunkLoader(player, (TileChunkLoader)tile, false));
				if(ID == GUI_CHUNKLOADER_FUELED)
					return new GuiAnchorFueled(new ContainerChunkLoader(player, (TileChunkLoader)tile, true));
				return null;
			}

			@Override
			public Object getServerGuiElement(int ID, EntityPlayer player, World world, int x, int y, int z) {
				TileEntity tile = world.getBlockTileEntity(x, y, z);
				if(ID == GUI_CHUNKLOADER)
					return new ContainerChunkLoader(player, (TileChunkLoader)tile, false);
				if(ID == GUI_CHUNKLOADER_FUELED)
					return new ContainerChunkLoader(player, (TileChunkLoader)tile, true);
				return null;
			}
		});

		APILocator.getNetManager().listen(this);

		{
			Configuration cfg = Config.config;

			if(!cfg.categories.containsKey("chunkloader.playerQuotaOverride")) {
				cfg.categories.put("chunkloader.playerQuotaOverride", new ConfigCategory("chunkloader.playerQuotaOverride"));
				Config.save();
			}
			playerQuotaOverride = cfg.categories.get("chunkloader.playerQuotaOverride");

			Fuels.setProperty(cfg.get(Configuration.CATEGORY_GENERAL, "chunkloader.fuels", ""));
		}

		commands.put("getquota", new Command() {
			@Override public String getUsage() {return "/dimanc getquota <username>";}
			@Override public void invoke(CommandUser cs, String[] args, int nextarg) {
				if(args.length <= nextarg) {
					cs.send("Not enough arguments.");
					return;
				}

				String player = args[nextarg++].toLowerCase();

				cs.send("Player "+player+" has used "+getCurQuota(player)+" of "+getMaxQuotaString(getMaxQuota(player))+" chunks.");
			}
		});

		commands.put("override", new Command() {
			@Override public String getUsage() {return "/dimanc override <username> {<quota>|unlimited|default} - sets a quota override for a specific player";}
			@Override public void invoke(CommandUser cs, String[] args, int nextarg) {
				if(args.length <= nextarg) {
					cs.send("Not enough arguments.");
					return;
				}
				String player = args[nextarg++].toLowerCase();
				if(args.length <= nextarg) {
					cs.send("Not enough arguments.");
					return;
				}

				String amtS = args[nextarg++];
				if(amtS.equalsIgnoreCase("default")) {
					playerQuotaOverride.remove(player);
					cs.send("\u00a7bOverride for "+player+" removed.");
				} else if(amtS.equalsIgnoreCase("unlimited")) {
					setQuotaOverride(player, "unlimited");
					cs.send("\u00a7b"+player+" now has unlimited chunks.");
				} else {
					try {
						int amt = Integer.parseInt(amtS);
						setQuotaOverride(player, String.valueOf(amt));
						cs.send("\u00a7b"+player+" now has "+amt+" chunks.");
					} catch(NumberFormatException e) {
						cs.send("\u00a7cSecond argument must be a number, 'default' or 'unlimited'.");
						return;
					}
				}

				EntityPlayer pl = getPlayer(player);
				if(pl != null)
					SidedProxy.instance.sendChat("\u00a7bYour chunk loading limit was changed by an admin.", pl);

				cs.send("\u00a7bNote that existing loaders will not be updated immediately.");
			}
		});

		requireFuel = Config.getBoolean("chunkloader.useFuel", false);
		allowFuelPiping = Config.getBoolean("chunkloader.allowFuelPiping", false) && requireFuel;

		//MAX_RADIUS = Config.getInt("chunkloader.maxRadius", MAX_RADIUS);

		if(requireFuel)
			Fuels.addCommands();

		String quotaType = Config.getString("chunkloader.quotaType",
				SidedProxy.instance.isDedicatedServer() ? "perplayer" : "unlimited",
						Configuration.CATEGORY_GENERAL,
						"Type of quota management to use for chunk loaders. Allowed values: "+Quota.getAllowedTypesString()
				);

		quota = Quota.createInstance(quotaType);

		showOtherPlayersLoaders = !Config.getBoolean("chunkloader.hideOtherPlayersLoadersInF9", SidedProxy.instance.isDedicatedServer());

		String logName = Config.getString("chunkloader.logFileName", !SidedProxy.instance.isDedicatedServer() ? "" : "dimensional-anchors.log", "logging", "Name of a file to log creation, deletion and editing of chunk loaders to. Blank for none.").trim();
		String listName = Config.getString("chunkloader.listFileName", "", "logging", "Name of a file to keep updated with a list of all active chunk loaders. Blank for none.").trim();

		if(!logName.isEmpty()) Logging.openLog(logName);
		if(!listName.isEmpty()) Logging.setList(listName);
	}

	public static int parseQuota(String s) throws NumberFormatException {
		if(s.equalsIgnoreCase("unlimited"))
			return Quota.UNLIMITED;
		else
			return Integer.parseInt(s);
	}

	public static String getMaxQuotaString(int i) {
		if(i == Quota.UNLIMITED)
			return "unlimited";
		else
			return String.valueOf(i);
	}



	@Override
	public boolean onTickInGame() {
		Set<World> toRemove = new HashSet<World>();
		for(Map.Entry<World, WorldInfo> e : worlds.entrySet()) {
			if(e.getValue() == null || !isWorldCurrent(e.getKey())) {
				toRemove.add(e.getKey());
				cli.onUnloadWorld(e.getValue());
				Logging.onUnloadWorld(e.getValue());
			} else {
				e.getValue().tick();
			}
		}
		for(World w : toRemove)
			worlds.remove(w);
		Logging.flushLog();
		return true;
	}

	public int getCurQuota(String player) {
		int r = 0;
		for(Map.Entry<World, WorldInfo> e : worlds.entrySet()) {
			int _this = e.getValue().getCurQuota(player);
			r += _this;
			//System.out.println(e.getKey()+" " + player + " -> " + _this);
		}
		return r;
	}

	public int getMaxQuota(String player) {
		player = player.toLowerCase();
		Property p = playerQuotaOverride.get(player);
		if(p != null)
			if(p.isIntValue())
				return p.getInt();
			else if(p.value.equals("unlimited"))
				return Quota.UNLIMITED;
		return quota.getMaxQuotaFor(player);
	}

	public boolean canAddQuota(String player, int i) {
		if(i <= 0)
			return true;
		int max = getMaxQuota(player);
		if(max == Quota.UNLIMITED)
			return true;
		return getCurQuota(player) + i <= max;
	}

	public static Iterable<WorldInfo> allWorlds() {
		return worlds.values();
	}

	private void setQuotaOverride(String key, String value) {
		if(playerQuotaOverride.containsKey(key)) {
			playerQuotaOverride.get(key).value = value;
		} else {
			playerQuotaOverride.put(key, new Property());
			playerQuotaOverride.get(key).value = value;
			playerQuotaOverride.get(key).setName(key);
		}
		Config.save();
	}

	@Override
	public String getChannel() {
		return CHANNEL;
	}

	@Override
	public IPacket createS2CPacket(byte id) {
		if(id == S2C_GUI_UPDATE)
			return new PacketGUIUpdate("", 0, 0, 0, false, false);
		if(id == S2C_DATA_RESPONSE)
			return new PacketShowChunksResponse();
		return null;
	}

	@Override
	public IPacket createC2SPacket(byte id) {
		if(id == C2S_DATA_REQUEST)
			return new PacketShowChunksRequest();
		return null;
	}
}
