package immibis.ars.beams;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.minecraft.src.AxisAlignedBB;
import net.minecraft.src.DamageSource;
import net.minecraft.src.Entity;
import net.minecraft.src.EntityFallingSand;
import net.minecraft.src.EntityItem;
import net.minecraft.src.EntityLiving;
import net.minecraft.src.EntityPainting;
import net.minecraft.src.EntityPlayer;
import net.minecraft.src.EntityTNTPrimed;
import net.minecraft.src.InventoryPlayer;
import net.minecraft.src.ItemStack;
import net.minecraft.src.NBTTagCompound;
import net.minecraft.src.Potion;
import net.minecraft.src.PotionEffect;
import net.minecraft.src.TileEntity;
import ic2.api.*;
import immibis.ars.mod_AdvancedRepulsionSystems;
import immibis.core.net.IPacket;
import immibis.core.net.OneTwoFiveNetworking;

public class TileTeslaCoil extends TileBeamEmitter implements IEnergySink {
	@Override protected void setupBeams() {}
	@Override public Object getOutput(int side) {return null;}
	
	/* Notes:
	 * At 32 eu/t
	 * 	skeletons die in 2 hits
	 *  zombies in 3
	 *  endermen in 4
	 * At 128 eu/t:
	 *  skeletons in 1
	 *  zombies in 2
	 *  endermen in 2
	 */
	
	private int filter_side = -1;
	
	@Override
	public void onBlockNeighbourChange() {
		super.onBlockNeighbourChange();
		filter_side = -1;
	}
	
	private static final int MAX_STORAGE = 5000;
	private static final int MIN_SHOT_EU = 500;
	private static final int MIN_INTERVAL = 1; // ticks
	private static final int DEFAULT_INTERVAL = 20; // ticks
	private static final int DEFAULT_ITEM_INTERVAL = 3; // ticks
	private static final double DEFAULT_RANGE = 10;
	
	private Set<Entity> currentTargets = null;
	private Entity currentTarget = null;
	private int ticksToNextShot = 0;
	private int eu_stored = 0;
	
	private boolean addedToENet = false;
	
	private UpgradeData upgrades;
	
	private double RANGE; // updated each shot from DEFAULT_RANGE and upgrades
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setInteger("energy", eu_stored);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		eu_stored = tag.getInteger("energy");
	}
	
	@Override
	public void updateEntity() {
		if(worldObj.isRemote)
			return;
		
		if(!addedToENet) {
			EnergyNet.getForWorld(worldObj).addTileEntity(this);
			addedToENet = true;
		}
		
		if(filter_side == -1) {
			for(int k = 0; k < 6; k++) {
				Object input = getInput(k);
				if(input instanceof EntityFilter) {
					filter_side = k;
					break;
				}
			}
		}
		
		if(ticksToNextShot <= 0) {
			if(eu_stored >= MIN_SHOT_EU) {
				EntityFilter filter;
				if(filter_side == -1)
					filter = null;
				else {
					Object filter_raw = getInput(filter_side);
					if(!(filter_raw instanceof EntityFilter))
						filter = null;
					else
						filter = (EntityFilter)filter_raw;
				}
				
				upgrades = new UpgradeData();
				for(int k = 0; k < 6; k++) {
					Object data = getInput(k);
					if(data instanceof UpgradeData) {
						upgrades.combine((UpgradeData)data);
					}
				}
				
				RANGE = DEFAULT_RANGE + upgrades.range;
				
				Set<Entity> targets = findAllPotentialTargets();
				if(filter != null)
					targets = filter.filter(targets);
				
				currentTargets = targets;
				
				if(targets != null && targets.size() > 0) {
					// Pick closest target
					Entity best = null;
					double best_dsq = Double.MAX_VALUE;
					for(Entity e : currentTargets) {
						double dsq = e.getDistanceSq(xCoord+0.5, yCoord+0.5, zCoord+0.5);
						if(dsq < best_dsq) {
							best_dsq = dsq;
							best = e;
						}
					}
					
					currentTarget = best;
					
					if(best != null)
						fireShot(best, eu_stored);
					
				} else
					currentTarget = null;
			}

		}
		
		ticksToNextShot--;
	}
	
	private static DamageSource damageSource = new DamageSource("ARSteslacoil") {};
	
	private static final int[] EMP_ARMOUR_PRIORITY = {
		2, // chestplate
		1, // legs
		3, // helmet
		0, // boots
	};
	
	private int getItemCollectionCost(EntityItem item) {
		double dx = item.posX - (xCoord + 0.5);
		double dy = item.posY - (yCoord + 0.5);
		double dz = item.posZ - (zCoord + 0.5);
		double dist_sq = dx*dx + dy*dy + dz*dz;
		
		// An item at 50 blocks will take 1000 EU
		return (int)(dist_sq * 0.4); 
	}
	
	private void fireShot(Entity at, int eu) {
		ticksToNextShot = Math.max(MIN_INTERVAL, (int)(DEFAULT_INTERVAL / upgrades.speed));
		
		if(at instanceof EntityItem) {
			if(eu > 384)
				eu = 384;
			ticksToNextShot = Math.min(ticksToNextShot, DEFAULT_ITEM_INTERVAL);
		}
		
		if(eu_stored < eu) {
			ticksToNextShot = 0;
			return;
		}

		eu_stored -= eu;
		
		
		
		if(at instanceof EntityPlayer && upgrades.emp != null && upgrades.emp.storage > 0) {
			int emp_eu = upgrades.emp.storage;
			
			InventoryPlayer inv = ((EntityPlayer)at).inventory;
			for(int k = 0; k < 4 && emp_eu > 0; k++) {
				ItemStack item = inv.armorInventory[EMP_ARMOUR_PRIORITY[k]];
				if(item != null)
					emp_eu -= ElectricItem.discharge(item, emp_eu, 9001, true, false);
			}
			
			upgrades.emp.storage = emp_eu;
		}
		
		
		
		int damage = (int)(Math.sqrt(eu / 5000.0) * 30);
		at.attackEntityFrom(damageSource, damage);
		
		if(upgrades.lootCollectors != null && at instanceof EntityItem && at.isDead) {
			EntityItem ei = (EntityItem)at;
			int origStackSize = ei.item.stackSize;
			for(TileLootCollector tlc : upgrades.lootCollectors)
				if(tlc.collectItem(ei))
					break;
			
			// may cause negative stored EU
			// since we don't check if there's enough first
			// (which we can't because we don't know whether the item
			// will be collected or not)
			if(ei.item.stackSize < origStackSize)
				eu_stored -= getItemCollectionCost(ei);
		}
		
		
		
		if(upgrades.potions != null && upgrades.potions.size() >= 1 && at instanceof EntityLiving) {
			TilePotionUpgrade tpu = upgrades.potions.iterator().next();
			for(PotionEffect pe : tpu.getEffect())
				((EntityLiving)at).addPotionEffect(new PotionEffect(pe));
		}
		
		
		
		if(!upgrades.suppressor) {
			// create bolt fx on clients
			AxisAlignedBB atBB = at.boundingBox;
			IPacket packet = new PacketBoltFX(
					xCoord + 0.5, yCoord + 0.5, zCoord + 0.5,
					(atBB.maxX + atBB.minX) / 2, (atBB.maxY + atBB.minY) / 2, (atBB.maxZ + atBB.minZ) / 2
				);
			List nearbyPlayers = worldObj.getEntitiesWithinAABB(EntityPlayer.class, AxisAlignedBB.getBoundingBox(xCoord, yCoord, zCoord, xCoord, yCoord, zCoord).offset(0.5, 0.5, 0.5).expand(100, 100, 100));
			for(EntityPlayer ep : (List<EntityPlayer>)nearbyPlayers)
				OneTwoFiveNetworking.send(mod_AdvancedRepulsionSystems.CHANNEL, packet, ep);
		}
	}
	
	private Set<Entity> findAllPotentialTargets() {
		double x = xCoord + 0.5;
		double y = yCoord + 0.5;
		double z = zCoord + 0.5;
		
		double rangesq = RANGE*RANGE;
		
		AxisAlignedBB aabb = AxisAlignedBB.getBoundingBox(
			x - RANGE,
			y - RANGE,
			z - RANGE,
			x + RANGE,
			y + RANGE,
			z + RANGE
		);
		
		Set<Entity> out = new HashSet<Entity>();
		for(Entity e : (List<Entity>)worldObj.getEntitiesWithinAABB(Entity.class, aabb)) {
			if(e instanceof EntityLiving && ((EntityLiving)e).deathTime > 0)
				continue; // never target dead mobs
			
			if(e instanceof EntityPainting || e instanceof EntityFallingSand || e instanceof EntityTNTPrimed)
				continue;
			
			if(e.getDistanceSq(x, y, z) <= rangesq)
				out.add(e);
		}
		
		return out;
	}
	
	@Override
	public void invalidate() {
		super.invalidate();
		if(addedToENet) {
			EnergyNet.getForWorld(worldObj).removeTileEntity(this);
			addedToENet = false;
		}
	}
	
	@Override
	public boolean acceptsEnergyFrom(TileEntity emitter, Direction direction) {
		return true;
	}
	
	@Override
	public boolean isAddedToEnergyNet() {
		return addedToENet;
	}
	
	@Override
	public boolean demandsEnergy() {
		return eu_stored < MAX_STORAGE;
	}
	
	@Override
	public int injectEnergy(Direction directionFrom, int amount) {
		if(eu_stored < MAX_STORAGE) {
			eu_stored += amount;
			return 0;
		}
		return amount;
	}
}
